(function (blocks, i18n, element, blockEditor, components) {

    const el = element.createElement;
    const __ = i18n.__;

    const MediaUpload       = blockEditor.MediaUpload;
    const InspectorControls = blockEditor.InspectorControls;
    const TextControl       = components.TextControl;
    const Button            = components.Button;

    // ------------------------------------------------------------
    // Helpers
    // ------------------------------------------------------------

    const isExternal = (url) =>
        /youtu\.be|youtube\.com|vimeo\.com/.test(url);

    const convertEmbed = (url) => {
        if (!url) return "";

        if (url.includes("youtube.com/watch?v=")) {
            return url.replace("watch?v=", "embed/");
        }
        if (url.includes("youtu.be/")) {
            return url.replace("youtu.be/", "youtube.com/embed/");
        }
        if (url.includes("vimeo.com/")) {
            return url.replace("vimeo.com/", "player.vimeo.com/video/");
        }
        return url;
    };

    // ------------------------------------------------------------
    // Block
    // ------------------------------------------------------------

    blocks.registerBlockType("mediageo/video-geo", {

        edit: function (props) {
            const attrs = props.attributes;
            const set   = props.setAttributes;

            const previewUrl = attrs.externalUrl || attrs.url;

            return [
                // ------------------------------------------------------------------
                // SIDEBAR : Title / Description / Licence / URL externe
                // ------------------------------------------------------------------
                el(
                    InspectorControls,
                    {},
                    el(TextControl, {
                        label: __("Titre", "mediageo"),
                        value: attrs.title,
                        onChange: v => set({ title: v })
                    }),
                    el(TextControl, {
                        label: __("Description", "mediageo"),
                        value: attrs.description,
                        onChange: v => set({ description: v })
                    }),
                    el(TextControl, {
                        label: __("Licence (URL)", "mediageo"),
                        value: attrs.license,
                        onChange: v => set({ license: v })
                    }),
                    el(TextControl, {
                        label: __("URL YouTube ou Vimeo", "mediageo"),
                        value: attrs.externalUrl,
                        placeholder: "https://youtube.com/... ou https://vimeo.com/...",
                        onChange: v => set({ externalUrl: v })
                    })
                ),

                // ------------------------------------------------------------------
                // Main editor preview
                // ------------------------------------------------------------------
                el(
                    "div",
                    { className: "mediageo-video-editor" },

                    // Preview
                    previewUrl
                        ? (
                            isExternal(previewUrl)
                                ? el("iframe", {
                                    src: convertEmbed(previewUrl),
                                    style: { width: "100%", height: "300px" }
                                })
                                : el("video", {
                                    src: previewUrl,
                                    controls: true,
                                    style: { width: "100%" }
                                })
                        )
                        : el("p", {}, __("Aucune vidéo sélectionnée ou URL externe.", "mediageo")),

                    // Upload button
                    el(MediaUpload, {
                        allowedTypes: ["video"],
                        value: attrs.videoId,
                        onSelect: (media) => {
                            set({
                                videoId: media.id,
                                url: media.url,
                                externalUrl: "" // reset external URL if a local file is chosen
                            });
                        },
                        render: ({ open }) =>
                            el(
                                Button,
                                { className: "button button-primary", onClick: open },
                                attrs.url
                                    ? __("Changer la vidéo locale", "mediageo")
                                    : __("Choisir une vidéo locale", "mediageo")
                            )
                    })
                )
            ];
        },

        save: function (props) {
            const attrs = props.attributes;

            const finalUrl  = attrs.externalUrl || attrs.url;
            const license   = attrs.license || "https://creativecommons.org/licenses/by-sa/4.0/";

            return el(
                "div",
                { className: "mediageo-video" },

                attrs.externalUrl
                    ? el("iframe", {
                        src: convertEmbed(attrs.externalUrl),
                        allowFullScreen: true
                    })
                    : el("video", { src: attrs.url, controls: true }),

                attrs.title ? el("h4", {}, attrs.title) : null,
                attrs.description ? el("p", {}, attrs.description) : null,

                el(
                    "p",
                    { className: "license" },
                    "Licence : ",
                    el("a", { href: license }, license)
                )
            );
        }
    });

})(window.wp.blocks, window.wp.i18n, window.wp.element, window.wp.blockEditor, window.wp.components);
