<?php
/**
 * Plugin Name: MediaGEO
 * Description: Blocs Gutenberg ImageGEO, VideoGEO et AudioGEO avec JSON-LD automatique + lightbox front.
 * Version: 2.1.9
 * Author: Erwan Tanguy - Ticoët
 * Text Domain: mediageo
 */

if (!defined('ABSPATH')) exit;

define('MEDIAGEO_PATH', plugin_dir_path(__FILE__));
define('MEDIAGEO_URL',  plugin_dir_url(__FILE__));

/**
 * --------------------------------------------------------------------------
 * 1. Enregistrement des blocs (avec render_callback)
 * --------------------------------------------------------------------------
 */
function mediageo_register_blocks() {

    $blocks = [
        'image-geo' => 'mediageo_render_image_geo',
        'video-geo' => 'mediageo_render_video_geo',
        'audio-geo' => 'mediageo_render_audio_geo'
    ];

    foreach ($blocks as $block => $callback) {

        register_block_type(
            MEDIAGEO_PATH . "blocks/$block",
            [
                'render_callback' => $callback, // très important
            ]
        );
    }
}
add_action('init', 'mediageo_register_blocks');


/**
 * --------------------------------------------------------------------------
 * 2. Rendu du bloc VIDEO + JSON-LD
 * --------------------------------------------------------------------------
 */
function mediageo_render_video_geo($attrs, $content) {

    if (empty($attrs['url']) && empty($attrs['externalUrl'])) {
        return '';
    }

    $title       = $attrs['title'] ?? '';
    $description = $attrs['description'] ?? '';
    $license     = $attrs['license'] ?? '';
    $url         = $attrs['externalUrl'] ?? $attrs['url'];

    // Détection embed (Youtube/Vimeo)
    if (!empty($attrs['externalUrl'])) {
        $embed = $attrs['externalUrl'];

        if (strpos($embed, 'youtube.com/watch?v=') !== false) {
            $embed = str_replace('watch?v=', 'embed/', $embed);
        }
        if (strpos($embed, 'youtu.be/') !== false) {
            $embed = str_replace('youtu.be/', 'youtube.com/embed/', $embed);
        }
        if (strpos($embed, 'vimeo.com/') !== false) {
            $embed = str_replace('vimeo.com/', 'player.vimeo.com/video/', $embed);
        }

        $html_video = '<iframe src="' . esc_url($embed) . '" allowfullscreen></iframe>';
    } else {
        $html_video = '<video src="' . esc_url($url) . '" controls></video>';
    }

    /**
     * JSON-LD VideoObject
     */
    $json = [
        "@context"      => "https://schema.org",
        "@type"         => "VideoObject",
        "name"          => esc_html($title),
        "description"   => esc_html($description),
        "contentUrl"    => esc_url($url),
        "license"       => esc_url($license),
    ];

    $json_ld = '<script type="application/ld+json">' . wp_json_encode($json, JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT) . '</script>';

    return '
        <div class="mediageo-video">
            ' . $html_video . '
            ' . ($title ? "<h4>".esc_html($title)."</h4>" : "") . '
            ' . ($description ? "<p>".esc_html($description)."</p>" : "") . '
            <p class="license">Licence : <a href="'.esc_url($license).'">'.esc_html($license).'</a></p>
        </div>
        ' . $json_ld;
}


/**
 * --------------------------------------------------------------------------
 * 3. Rendu du bloc IMAGE + JSON-LD
 * --------------------------------------------------------------------------
 */
function mediageo_render_image_geo($attrs, $content) {

    if (empty($attrs['url'])) return '';

    $title       = $attrs['alt'] ?? '';
    $description = $attrs['description'] ?? '';
    $caption	 = $attrs['caption'] ?? '';
    $license     = $attrs['license'] ?? '';
    $url         = $attrs['url'];
	$fullUrl	 = $attrs['fullUrl'];

    $html = '<figure class="wp-block-mediageo-image-geo">
				<a href="'.esc_url($fullUrl).'" class="geo-lightbox" data-geo-src="'.esc_url($fullUrl).'">
                	<img src="'.esc_url($url).'" alt="'.esc_attr($description).'">
				</a>
                ' . ($caption ? "<figcaption>".esc_html($caption)."</figcaption>" : "") . '
            </figure>';

    $json = [
        "@context"      => "https://schema.org",
        "@type"         => "ImageObject",
        "name"          => esc_html($title),
        "description"   => esc_html($description),
        "caption"	=> esc_html($caption),
        "contentUrl"    => esc_url($url),
        "license"       => esc_url($license),
    ];

    return $html . '<script type="application/ld+json">'.wp_json_encode($json, JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT).'</script>';
}


/**
 * --------------------------------------------------------------------------
 * 4. Rendu du bloc AUDIO + JSON-LD
 * --------------------------------------------------------------------------
 */
function mediageo_render_audio_geo($attrs, $content) {

    if (empty($attrs['url'])) return '';

    $title       = $attrs['title'] ?? '';
    $description = $attrs['description'] ?? '';
    $license     = $attrs['license'] ?? '';
    $url         = $attrs['url'];
	$transcript	 = $attrs['transcript'];

    $html = '<div class="mediageo-audio">
                <audio src="'.esc_url($url).'" controls></audio>
                ' . ($title ? "<h4>".esc_html($title)."</h4>" : "") . '
                ' . ($description ? "<p>".esc_html($description)."</p>" : "") . '
				' . ($transcript ? "<blockquote><b>Transcription :</b><br>".esc_html($transcript)."</blockquote>" : "") . '
            </div>';

    $json = [
        "@context"      => "https://schema.org",
        "@type"         => "AudioObject",
        "name"          => esc_html($title),
        "description"   => esc_html($description),
		"transcript"	=> esc_html($transcript),
        "contentUrl"    => esc_url($url),
        "license"       => esc_url($license),
    ];

    return $html . '<script type="application/ld+json">'.wp_json_encode($json, JSON_UNESCAPED_SLASHES|JSON_PRETTY_PRINT).'</script>';
}


/**
 * --------------------------------------------------------------------------
 * 5. Styles + JS globaux
 * --------------------------------------------------------------------------
 */
function mediageo_global_assets() {

    // Styles
    wp_enqueue_style(
        'mediageo-global-style',
        MEDIAGEO_URL . 'style.css',
        [],
        filemtime(MEDIAGEO_PATH . 'style.css')
    );

    // Lightbox JS
    wp_enqueue_script(
        'mediageo-lightbox',
        MEDIAGEO_URL . 'lightbox.js',
        [],
        filemtime(MEDIAGEO_PATH . 'lightbox.js'),
        true
    );
}
add_action('wp_enqueue_scripts', 'mediageo_global_assets');
