(function (wp) {
    const { blocks, i18n, element, blockEditor, components } = wp;
    const { createElement: el, useEffect } = element;
    const { __ } = i18n;
    const { registerBlockType } = blocks;
    const {
        InspectorControls,
        MediaUpload,
        useBlockProps,
    } = blockEditor;
    const {
        PanelBody,
        TextControl,
        Button,
        ExternalLink,
    } = components;

    // Helpers
    const isExternal = (url) =>
        /youtu\.be|youtube\.com|vimeo\.com/.test(url);

    const convertEmbed = (url) => {
        if (!url) return "";
        if (url.includes("youtube.com/watch?v=")) {
            return url.replace("watch?v=", "embed/");
        }
        if (url.includes("youtu.be/")) {
            return url.replace("youtu.be/", "youtube.com/embed/");
        }
        if (url.includes("vimeo.com/")) {
            return url.replace("vimeo.com/", "player.vimeo.com/video/");
        }
        return url;
    };

    registerBlockType("mediageo/video-geo", {
        apiVersion: 3,
        title: __("Video GEO", "mediageo"),
        category: "media",
        icon: "format-video",
        description: __("Intègre une vidéo locale ou un lien externe YouTube/Vimeo, avec métadonnées GEO.", "mediageo"),

        attributes: {
            videoId: { type: "number" },
            url: { type: "string" },
            externalUrl: { type: "string" },
            title: { type: "string" },
            description: { type: "string" },
            license: { type: "string", default: "https://creativecommons.org/licenses/by-sa/4.0/" }
        },

        edit: function (props) {
            const { attributes, setAttributes, clientId } = props;
            const { videoId, url, externalUrl, title, description, license } = attributes;

            const previewUrl = externalUrl || url;

            // Use Block Props to ensure the block is selectable / deletable / draggable
            const blockProps = useBlockProps({
                className: "mediageo-video-block",
                "data-client-id": clientId
            });

            // When a media is selected, fill attributes (url, id) and try to populate title/description
            const onSelectMedia = (media) => {
                if (!media) return;
                setAttributes({
                    videoId: media.id || null,
                    url: media.url || "",
                    externalUrl: "", // reset external URL when choosing local file
                    // prefer existing title/description if already set by user
                    title: attributes.title || media.title || "",
                    description: attributes.description || media.description || ""
                });
            };

            // Helper to clear selection (remove urls)
            const clearMedia = () => {
                setAttributes({
                    videoId: null,
                    url: "",
                    externalUrl: "",
                    title: "",
                    description: ""
                });
            };

            return [
                // Inspector controls (sidebar)
                el(
                    InspectorControls,
                    {},
                    el(
                        PanelBody,
                        { title: __("Métadonnées vidéo", "mediageo"), initialOpen: true },
                        el(TextControl, {
                            label: __("Titre", "mediageo"),
                            value: title || "",
                            onChange: v => setAttributes({ title: v })
                        }),
                        el(TextControl, {
                            label: __("Description", "mediageo"),
                            value: description || "",
                            onChange: v => setAttributes({ description: v })
                        }),
                        el(TextControl, {
                            label: __("Licence (URL)", "mediageo"),
                            value: license || "",
                            onChange: v => setAttributes({ license: v })
                        }),
                        el("div", { style: { marginTop: "8px" } },
                            el("strong", {}, __("Ou URL externe (YouTube / Vimeo) :", "mediageo")),
                            el(TextControl, {
                                placeholder: "https://youtube.com/...",
                                value: externalUrl || "",
                                onChange: v => setAttributes({ externalUrl: v })
                            }),
                            el("small", { style: { display: "block", marginTop: "6px", color: "#666" } },
                                __("Si vous renseignez une URL externe, la vidéo locale sera ignorée.", "mediageo")
                            )
                        )
                    )
                ),

                // Main editor UI
                el(
                    "div",
                    blockProps,
                    // Preview
                    el(
                        "div",
                        { className: "mediageo-video-editor" },
                        previewUrl
                            ? (
                                isExternal(previewUrl)
                                    ? el("iframe", {
                                        src: convertEmbed(previewUrl),
                                        style: { width: "100%", height: "300px", border: 0 },
                                        frameBorder: 0,
                                        allowFullScreen: true
                                    })
                                    : el("video", {
                                        src: previewUrl,
                                        controls: true,
                                        style: { width: "100%", maxHeight: "400px" }
                                    })
                            )
                            : el("p", {}, __("Aucune vidéo sélectionnée ou URL externe.", "mediageo"))
                    ),

                    // Controls under preview
                    el(
                        "div",
                        { style: { marginTop: "12px", display: "flex", gap: "8px", alignItems: "center" } },

                        // MediaUpload button (local upload)
                        el(
                            MediaUpload,
                            {
                                allowedTypes: ["video"],
                                value: videoId,
                                onSelect: onSelectMedia,
                                render: ({ open }) => el(
                                    Button,
                                    {
                                        isPrimary: true,
                                        onClick: open
                                    },
                                    url ? __("Changer la vidéo locale", "mediageo") : __("Choisir une vidéo locale", "mediageo")
                                )
                            }
                        ),

                        // Quick clear button
                        el(
                            Button,
                            {
                                isSecondary: true,
                                onClick: clearMedia,
                                disabled: !previewUrl
                            },
                            __("Supprimer la sélection", "mediageo")
                        ),

                        // If externalUrl present, show small hint
                        externalUrl ? el("div", { style: { marginLeft: "12px", fontSize: "13px", color: "#444" } },
                            el("span", {}, __("URL externe active", "mediageo")),
                            el("span", { style: { marginLeft: "8px" } },
                                el("a", { href: externalUrl, target: "_blank", rel: "noopener noreferrer" }, __("Voir", "mediageo"))
                            )
                        ) : null
                    ),

                    // Display metadata summary (editable via InspectorControls)
                    el(
                        "div",
                        { className: "mediageo-video-meta", style: { marginTop: "12px" } },
                        title ? el("strong", {}, title) : null,
                        title && description ? el("br") : null,
                        description ? el("span", {}, description) : null,
                        el("div", { style: { marginTop: "6px", color: "#666", fontSize: "13px" } },
                            el("span", {}, __("Licence :", "mediageo") + " "),
                            el("a", { href: license || "#", target: "_blank", rel: "noopener noreferrer" }, license)
                        )
                    )
                )
            ];
        },

        save: function (props) {
            const attrs = props.attributes || {};
            const finalUrl = attrs.externalUrl || attrs.url;
            const license = attrs.license || "https://creativecommons.org/licenses/by-sa/4.0/";

            // Use block props for front markup too
            const blockProps = { className: "mediageo-video" };

            return el(
                "div",
                blockProps,
                attrs.externalUrl
                    ? el("iframe", {
                        src: convertEmbed(attrs.externalUrl),
                        allowFullScreen: true,
                        frameBorder: 0
                    })
                    : el("video", { src: attrs.url, controls: true }),

                attrs.title ? el("h4", {}, attrs.title) : null,
                attrs.description ? el("p", {}, attrs.description) : null,

                el("p", { className: "license" },
                    "Licence : ",
                    el("a", { href: license }, license)
                )
            );
        }
    });

})(window.wp);
