<?php
/**
 * schema-organization.php - V0.3.0
 * Génère l'entité Organization principale du site
 * 
 * V0.3.0 - CORRECTION FINALE :
 * - Ne génère l'Organization QUE si aucune Organization EAS n'existe
 * - Évite les doublons Organization
 */

if (!defined('ABSPATH')) {
    exit;
}

add_action('wp_head', function () {
    
    // ✅ VÉRIFICATION : Une Organization EAS existe-t-elle déjà ?
    $eas_organizations = get_posts([
        'post_type' => 'entity',
        'posts_per_page' => 1,
        'post_status' => 'publish',
        'tax_query' => [
            [
                'taxonomy' => 'entity_type',
                'field' => 'name',
                'terms' => 'Organization',
            ],
        ],
    ]);
    
    // Si une Organization EAS existe, ne rien générer (éviter le doublon)
    if (!empty($eas_organizations)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('GEO Entities: Organization EAS détectée, skip génération GEO Organization');
        }
        return; // ✅ Sortir maintenant
    }
    
    // ✅ Sinon, générer l'Organization GEO (mode fallback)
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('GEO Entities: Aucune Organization EAS, génération Organization GEO');
    }

    $organization = [
        "@type"    => "Organization",
        "@id"      => geo_entity_id('organization'),
        "name"     => get_bloginfo('name'),
        "url"      => home_url('/'),
    ];

    // Description
    $description = get_bloginfo('description');
    if (!empty($description)) {
        $organization['description'] = $description;
    }

    // Logo
    $custom_logo_id = get_theme_mod('custom_logo');
    if ($custom_logo_id) {
        $logo_url = wp_get_attachment_image_url($custom_logo_id, 'full');
        if ($logo_url) {
            $organization['logo'] = $logo_url;
        }
    }

    // Liens sociaux
    $social_links = [];
    
    $facebook = get_option('geo_social_facebook');
    $twitter = get_option('geo_social_twitter');
    $linkedin = get_option('geo_social_linkedin');
    $instagram = get_option('geo_social_instagram');
    $youtube = get_option('geo_social_youtube');
    
    if ($facebook) $social_links[] = $facebook;
    if ($twitter) $social_links[] = $twitter;
    if ($linkedin) $social_links[] = $linkedin;
    if ($instagram) $social_links[] = $instagram;
    if ($youtube) $social_links[] = $youtube;
    
    if (!empty($social_links)) {
        $organization['sameAs'] = $social_links;
    }

    // Email
    $contact_email = get_option('geo_contact_email', get_option('admin_email'));
    if ($contact_email) {
        $organization['email'] = $contact_email;
    }

    // Téléphone
    $phone = get_option('geo_contact_phone');
    if ($phone) {
        $organization['telephone'] = $phone;
    }

    // Adresse
    $address_street = get_option('geo_address_street');
    $address_city = get_option('geo_address_city');
    $address_postal = get_option('geo_address_postal');
    $address_country = get_option('geo_address_country');
    
    if ($address_street || $address_city) {
        $organization['address'] = [
            "@type" => "PostalAddress"
        ];
        
        if ($address_street) $organization['address']['streetAddress'] = $address_street;
        if ($address_city) $organization['address']['addressLocality'] = $address_city;
        if ($address_postal) $organization['address']['postalCode'] = $address_postal;
        if ($address_country) $organization['address']['addressCountry'] = $address_country;
    }

    // Enregistrer dans le registre
    geo_register_entity($organization);

    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('GEO Entities: Organization GEO enregistrée - ' . $organization['name']);
    }

}, 20);

/**
 * ✅ Génère le JSON-LD complet à la fin
 * Priorité 100 : APRÈS tous les enregistrements
 */
add_action('wp_head', function () {
    
    $entities = geo_get_entities();
    
    if (empty($entities)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('GEO Entities: Aucune entité à afficher');
        }
        return;
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log(sprintf('GEO Entities: Génération JSON-LD pour %d entité(s)', count($entities)));
    }
    
    // Appliquer le filtre de nettoyage
    $entities = apply_filters('geo_entities_before_output', $entities);
    
    if (empty($entities)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('GEO Entities: Toutes les entités rejetées par le filtre');
        }
        return;
    }
    
    // Convertir en tableau indexé
    $graph = array_values($entities);
    
    // ✅ Nettoyage final
    foreach ($graph as $key => &$entity) {
        // Supprimer @context si présent
        if (isset($entity['@context'])) {
            unset($entity['@context']);
        }
        
        // Rejeter les @type invalides
        if (isset($entity['@type']) && in_array($entity['@type'], ['worksFor', 'memberOf'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log(sprintf(
                    '❌ GEO Entities: Entité avec @type invalide "%s" rejetée',
                    $entity['@type']
                ));
            }
            unset($graph[$key]);
        }
    }
    
    // Retirer les vides et ré-indexer
    $graph = array_filter($graph);
    $graph = array_values($graph);
    
    if (empty($graph)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('GEO Entities: Aucune entité valide après nettoyage');
        }
        return;
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log(sprintf('GEO Entities: %d entité(s) valides à afficher', count($graph)));
        
        // Log des types d'entités
        $types_count = [];
        foreach ($graph as $entity) {
            $type = $entity['@type'] ?? 'Unknown';
            $types_count[$type] = ($types_count[$type] ?? 0) + 1;
        }
        foreach ($types_count as $type => $count) {
            error_log(sprintf('  - %s: %d', $type, $count));
        }
    }
    
    // Afficher le JSON-LD
    echo "\n" . '<script type="application/ld+json">' . "\n";
    echo wp_json_encode(
        [
            '@context' => 'https://schema.org',
            '@graph' => $graph,
        ],
        JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT
    );
    echo "\n" . '</script>' . "\n";
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('GEO Entities: JSON-LD généré avec succès');
    }
    
}, 100);