<?php
/**
 * llms-generator.php
 * Génère automatiquement le fichier llms.txt à la racine du site
 * 
 * @see https://llmstxt.org/ pour la documentation du format
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Génère le contenu du fichier llms.txt
 */
function geo_generate_llms_content(): string {
    
    $site_name = get_bloginfo('name');
    $site_description = get_bloginfo('description');
    $site_url = home_url('/');
    
    // En-tête
    $content = "# $site_name\n\n";
    
    if (!empty($site_description)) {
        $content .= "> $site_description\n\n";
    }
    
    // Informations de base
    $content .= "## À propos\n\n";
    $content .= "Site web : $site_url\n";
    
    // Organization
    $org_email = get_option('geo_contact_email');
    $org_phone = get_option('geo_contact_phone');
    
    if ($org_email) {
        $content .= "Contact : $org_email\n";
    }
    if ($org_phone) {
        $content .= "Téléphone : $org_phone\n";
    }
    
    $content .= "\n";
    
    // Réseaux sociaux
    $socials = [];
    if ($fb = get_option('geo_social_facebook')) $socials['Facebook'] = $fb;
    if ($tw = get_option('geo_social_twitter')) $socials['Twitter'] = $tw;
    if ($li = get_option('geo_social_linkedin')) $socials['LinkedIn'] = $li;
    if ($ig = get_option('geo_social_instagram')) $socials['Instagram'] = $ig;
    if ($yt = get_option('geo_social_youtube')) $socials['YouTube'] = $yt;
    
    if (!empty($socials)) {
        $content .= "## Réseaux sociaux\n\n";
        foreach ($socials as $platform => $url) {
            $content .= "- $platform : $url\n";
        }
        $content .= "\n";
    }
    
    // Articles récents
    $recent_posts = get_posts([
        'post_type' => 'post',
        'posts_per_page' => get_option('geo_llms_posts_count', 20),
        'post_status' => 'publish',
        'orderby' => 'date',
        'order' => 'DESC',
    ]);
    
    if (!empty($recent_posts)) {
        $content .= "## Articles récents\n\n";
        
        foreach ($recent_posts as $post) {
            $title = get_the_title($post);
            $url = get_permalink($post);
            $excerpt = wp_strip_all_tags(get_the_excerpt($post));
            $date = get_the_date('Y-m-d', $post);
            
            $content .= "### $title\n";
            $content .= "URL : $url\n";
            $content .= "Date : $date\n";
            
            if (!empty($excerpt)) {
                $content .= "Résumé : $excerpt\n";
            }
            
            $content .= "\n";
        }
    }
    
    // Pages principales
    $main_pages = get_pages([
        'sort_column' => 'menu_order',
        'number' => 10,
    ]);
    
    if (!empty($main_pages)) {
        $content .= "## Pages principales\n\n";
        
        foreach ($main_pages as $page) {
            $title = get_the_title($page);
            $url = get_permalink($page);
            
            $content .= "- $title : $url\n";
        }
        
        $content .= "\n";
    }
    
    // Entités (si Entity Authority Signals est actif)
    if (post_type_exists('entity')) {
        $entities = get_posts([
            'post_type' => 'entity',
            'posts_per_page' => 50,
            'post_status' => 'publish',
        ]);
        
        if (!empty($entities)) {
            $content .= "## Entités référencées\n\n";
            
            // Grouper par type
            $entities_by_type = [];
            
            foreach ($entities as $entity_post) {
                $terms = get_the_terms($entity_post->ID, 'entity_type');
                $type = !empty($terms) ? $terms[0]->name : 'Autre';
                
                if (!isset($entities_by_type[$type])) {
                    $entities_by_type[$type] = [];
                }
                
                $entities_by_type[$type][] = [
                    'name' => get_the_title($entity_post),
                    'url' => get_permalink($entity_post),
                ];
            }
            
            foreach ($entities_by_type as $type => $entities_list) {
                $content .= "### $type\n\n";
                
                foreach ($entities_list as $entity) {
                    $content .= "- {$entity['name']} : {$entity['url']}\n";
                }
                
                $content .= "\n";
            }
        }
    }
    
    // Informations techniques
    $content .= "## Informations techniques\n\n";
    $content .= "- Format : llms.txt v1.0\n";
    $content .= "- Généré le : " . date('Y-m-d H:i:s') . "\n";
    $content .= "- CMS : WordPress " . get_bloginfo('version') . "\n";
    $content .= "- Plugin : GEO Entities Core\n";
    
    return $content;
}

/**
 * Écrit le fichier llms.txt à la racine du site
 */
function geo_write_llms_file(): bool {
    
    $content = geo_generate_llms_content();
    
    // Chemin du fichier
    $file_path = ABSPATH . 'llms.txt';
    
    // Écriture du fichier
    $result = file_put_contents($file_path, $content);
    
    if ($result === false) {
        return false;
    }
    
    // Log
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('GEO Entities: llms.txt généré avec succès');
    }
    
    return true;
}

/**
 * Génère llms.txt automatiquement lors de la publication d'un article
 */
add_action('save_post', function($post_id, $post) {
    
    // Vérifier si la génération auto est activée
    if (!get_option('geo_llms_auto_generate', false)) {
        return;
    }
    
    // Uniquement pour les articles et pages
    if (!in_array($post->post_type, ['post', 'page', 'entity'])) {
        return;
    }
    
    // Uniquement lors de la publication
    if ($post->post_status !== 'publish') {
        return;
    }
    
    // Éviter les révisions
    if (wp_is_post_revision($post_id)) {
        return;
    }
    
    // Générer le fichier
    geo_write_llms_file();
    
}, 10, 2);

/**
 * Ajoute un lien vers llms.txt dans le <head>
 */
add_action('wp_head', function() {
    
    if (!get_option('geo_llms_add_link', true)) {
        return;
    }
    
    $llms_url = home_url('/llms.txt');
    
    echo '<link rel="llms" href="' . esc_url($llms_url) . '" />' . "\n";
    
}, 1);

/**
 * Gestion AJAX pour générer manuellement le fichier
 */
add_action('wp_ajax_geo_generate_llms', function() {
    
    check_ajax_referer('geo_llms_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Permission refusée']);
    }
    
    $success = geo_write_llms_file();
    
    if ($success) {
        wp_send_json_success([
            'message' => 'Fichier llms.txt généré avec succès !',
            'url' => home_url('/llms.txt'),
        ]);
    } else {
        wp_send_json_error([
            'message' => 'Erreur lors de la génération du fichier. Vérifiez les permissions d\'écriture.',
        ]);
    }
});

/**
 * Ajoute la page de configuration llms.txt dans l'admin
 */
add_action('admin_menu', function() {
    
    // Vérifier que le menu parent existe
    global $menu;
    
    $parent_exists = false;
    foreach ($menu as $item) {
        if (isset($item[2]) && $item[2] === 'geo-entity-audit') {
            $parent_exists = true;
            break;
        }
    }
    
    if ($parent_exists) {
        add_submenu_page(
            'geo-entity-audit',
            'Fichier llms.txt',
            '📄 llms.txt',
            'manage_options',
            'geo-llms-generator',
            'geo_render_llms_page'
        );
    }
    
}, 20);

/**
 * Page de configuration llms.txt
 */
function geo_render_llms_page() {
    
    // Sauvegarde des options
    if (isset($_POST['geo_llms_save'])) {
        check_admin_referer('geo_llms_options');
        
        update_option('geo_llms_auto_generate', isset($_POST['geo_llms_auto_generate']));
        update_option('geo_llms_add_link', isset($_POST['geo_llms_add_link']));
        update_option('geo_llms_posts_count', intval($_POST['geo_llms_posts_count']));
        
        echo '<div class="notice notice-success"><p>Options enregistrées avec succès !</p></div>';
    }
    
    // Vérifier si le fichier existe
    $file_path = ABSPATH . 'llms.txt';
    $file_exists = file_exists($file_path);
    $file_url = home_url('/llms.txt');
    
    $auto_generate = get_option('geo_llms_auto_generate', false);
    $add_link = get_option('geo_llms_add_link', true);
    $posts_count = get_option('geo_llms_posts_count', 20);
    
    ?>
    <div class="wrap">
        <h1>📄 Générateur llms.txt</h1>
        
        <p class="description">
            Le fichier <code>llms.txt</code> aide les moteurs IA (ChatGPT, Perplexity, Claude) à mieux comprendre et indexer votre site.
            <a href="https://llmstxt.org/" target="_blank">En savoir plus sur llms.txt</a>
        </p>
        
        <!-- Statut du fichier -->
        <div class="card" style="margin: 20px 0;">
            <h2>📊 Statut</h2>
            
            <?php if ($file_exists): ?>
                <div class="notice notice-success inline">
                    <p>
                        ✅ Le fichier <code>llms.txt</code> existe et est accessible.
                    </p>
                </div>
                
                <p>
                    <strong>URL :</strong> 
                    <a href="<?php echo esc_url($file_url); ?>" target="_blank">
                        <?php echo esc_html($file_url); ?>
                    </a>
                </p>
                
                <p>
                    <strong>Dernière modification :</strong>
                    <?php echo date('d/m/Y H:i:s', filemtime($file_path)); ?>
                </p>
                
                <p>
                    <strong>Taille :</strong>
                    <?php echo size_format(filesize($file_path)); ?>
                </p>
                
            <?php else: ?>
                <div class="notice notice-warning inline">
                    <p>
                        ⚠️ Le fichier <code>llms.txt</code> n'existe pas encore.
                    </p>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Génération manuelle -->
        <div class="card" style="margin: 20px 0;">
            <h2>🔧 Générer le fichier</h2>
            
            <p>
                Cliquez sur le bouton ci-dessous pour générer ou mettre à jour le fichier <code>llms.txt</code>.
            </p>
            
            <p>
                <button type="button" id="geo-generate-llms" class="button button-primary button-hero">
                    🚀 Générer llms.txt
                </button>
            </p>
            
            <div id="geo-llms-result" style="margin-top: 15px;"></div>
        </div>
        
        <!-- Options -->
        <form method="post" action="">
            <?php wp_nonce_field('geo_llms_options'); ?>
            
            <div class="card" style="margin: 20px 0;">
                <h2>⚙️ Options</h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="geo_llms_auto_generate">Génération automatique</label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" 
                                       name="geo_llms_auto_generate" 
                                       id="geo_llms_auto_generate"
                                       value="1"
                                       <?php checked($auto_generate); ?> />
                                Régénérer automatiquement le fichier lors de la publication d'un article/page
                            </label>
                            <p class="description">
                                Recommandé pour garder le fichier à jour en permanence.
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="geo_llms_add_link">Lien dans le &lt;head&gt;</label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" 
                                       name="geo_llms_add_link" 
                                       id="geo_llms_add_link"
                                       value="1"
                                       <?php checked($add_link); ?> />
                                Ajouter <code>&lt;link rel="llms" href="/llms.txt" /&gt;</code> dans le &lt;head&gt;
                            </label>
                            <p class="description">
                                Aide les IA à découvrir automatiquement le fichier.
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="geo_llms_posts_count">Nombre d'articles</label>
                        </th>
                        <td>
                            <input type="number" 
                                   name="geo_llms_posts_count" 
                                   id="geo_llms_posts_count"
                                   value="<?php echo esc_attr($posts_count); ?>"
                                   min="5"
                                   max="100"
                                   class="small-text" />
                            <p class="description">
                                Nombre d'articles récents à inclure dans le fichier (5 à 100).
                            </p>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <button type="submit" name="geo_llms_save" class="button button-primary">
                        💾 Enregistrer les options
                    </button>
                </p>
            </div>
        </form>
        
        <!-- Aperçu -->
        <?php if ($file_exists): ?>
            <div class="card" style="margin: 20px 0;">
                <h2>👁️ Aperçu du fichier</h2>
                
                <textarea readonly style="width: 100%; height: 400px; font-family: monospace; font-size: 12px; padding: 10px;"><?php 
                    echo esc_textarea(file_get_contents($file_path)); 
                ?></textarea>
            </div>
        <?php endif; ?>
        
        <!-- Informations -->
        <div class="card" style="margin: 20px 0; background: #e7f3ff; border-left: 4px solid #2196f3;">
            <h2>💡 Qu'est-ce que llms.txt ?</h2>
            
            <p>
                Le fichier <code>llms.txt</code> est un standard émergent qui permet aux grands modèles de langage (LLMs) 
                comme ChatGPT, Claude, ou Perplexity de mieux comprendre la structure et le contenu de votre site.
            </p>
            
            <h3>Contenu généré automatiquement :</h3>
            <ul>
                <li>✅ Informations de base (nom, description, contact)</li>
                <li>✅ Réseaux sociaux</li>
                <li>✅ Articles récents avec résumés</li>
                <li>✅ Pages principales</li>
                <li>✅ Entités référencées (si EAS actif)</li>
            </ul>
            
            <h3>Avantages pour le GEO :</h3>
            <ul>
                <li>🎯 Meilleure indexation par les IA</li>
                <li>🎯 Citations plus précises de votre contenu</li>
                <li>🎯 Découverte facilitée de vos articles</li>
                <li>🎯 Contexte enrichi pour les moteurs IA</li>
            </ul>
            
            <p>
                <a href="https://llmstxt.org/" target="_blank" class="button">
                    📚 Documentation officielle llms.txt
                </a>
            </p>
        </div>
    </div>
    
    <script>
    jQuery(document).ready(function($) {
        $('#geo-generate-llms').on('click', function() {
            var $button = $(this);
            var $result = $('#geo-llms-result');
            
            $button.prop('disabled', true).text('⏳ Génération en cours...');
            $result.html('');
            
            $.ajax({
                url: ajaxurl,
                method: 'POST',
                data: {
                    action: 'geo_generate_llms',
                    nonce: '<?php echo wp_create_nonce('geo_llms_nonce'); ?>'
                },
                success: function(response) {
                    if (response.success) {
                        $result.html('<div class="notice notice-success inline"><p>✅ ' + response.data.message + '</p><p><a href="' + response.data.url + '" target="_blank">Voir le fichier</a></p></div>');
                        
                        // Recharger la page après 2 secondes pour afficher l'aperçu
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $result.html('<div class="notice notice-error inline"><p>❌ ' + response.data.message + '</p></div>');
                    }
                },
                error: function() {
                    $result.html('<div class="notice notice-error inline"><p>❌ Erreur de communication avec le serveur.</p></div>');
                },
                complete: function() {
                    $button.prop('disabled', false).text('🚀 Générer llms.txt');
                }
            });
        });
    });
    </script>
    <?php
}