<?php
/**
 * entity-audit.php - VERSION AMÉLIORÉE
 * Audit de cohérence des entités GEO
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Lance l'audit de cohérence des entités GEO
 * 
 * @return array Résultats de l'audit (errors, warnings, info, entities)
 */
function geo_run_entity_audit(): array {

    // Forcer le chargement des entités en simulant une page
    // (car wp_head n'est pas exécuté dans l'admin)
    do_action('wp_head');

    $entities = geo_get_entities();
    $count = count($entities);
    
    $results = [
        'errors'   => [],
        'warnings' => [],
        'info'     => [],
        'entities' => $entities,
        'count'    => $count,
        'sources'  => [] // Nouvea: tracer la provenance des entités
    ];
    
    // Détecter la source des entités
    $geo_native = 0;
    $eas_bridge = 0;
    
    foreach ($entities as $entity) {
        $id = $entity['@id'] ?? '';
        
        // Entités d'Entity Authority Signals ont un @id avec #entity-XXX
        if (strpos($id, '#entity-') !== false && preg_match('/#entity-(\d+)/', $id, $matches)) {
            $post_id = intval($matches[1]);
            if (get_post_type($post_id) === 'entity') {
                $eas_bridge++;
                continue;
            }
        }
        
        $geo_native++;
    }
    
    // Info sur les sources
    if ($geo_native > 0) {
        $results['sources'][] = sprintf('GEO Entities natif : %d entité(s)', $geo_native);
    }
    if ($eas_bridge > 0) {
        $results['sources'][] = sprintf('Entity Authority Signals : %d entité(s)', $eas_bridge);
    }

    // Info générale
    $results['info'][] = sprintf('Nombre total d\'entités détectées : %d', $count);

    if ($count === 0) {
        $results['errors'][] = 'Aucune entité n\'a été enregistrée. Vérifiez que les hooks wp_head sont bien exécutés.';
        
        // Vérifier si Entity Authority Signals est installé
        if (post_type_exists('entity')) {
            $eas_count = wp_count_posts('entity');
            if ($eas_count->publish > 0) {
                $results['warnings'][] = sprintf(
                    'Le plugin Entity Authority Signals est détecté avec %d entité(s) publiée(s), mais elles ne sont pas enregistrées dans GEO Entities. Installez le plugin "GEO Entities + EAS Bridge" pour les détecter.',
                    $eas_count->publish
                );
            }
        }
        
        return $results;
    }

    // --- A. Vérifier l'Organization ---
    $organizations = array_filter($entities, function ($entity) {
        return ($entity['@type'] ?? '') === 'Organization';
    });

    $org_count = count($organizations);
    $results['info'][] = sprintf('Entités Organization trouvées : %d', $org_count);

    if ($org_count === 0) {
        $results['errors'][] = 'Aucune entité Organization détectée. Votre site doit avoir une organisation principale.';
    } elseif ($org_count > 1) {
        $results['errors'][] = sprintf(
            'Plusieurs entités Organization détectées (%d). Il ne devrait y en avoir qu\'une seule (collision d\'autorité).',
            $org_count
        );
    } else {
        $org = array_values($organizations)[0];
        
        // Vérifications de l'Organization
        if (empty($org['name'])) {
            $results['errors'][] = 'L\'Organization n\'a pas de nom.';
        }
        
        if (empty($org['url'])) {
            $results['warnings'][] = 'L\'Organization n\'a pas d\'URL.';
        }
        
        if (empty($org['description'])) {
            $results['warnings'][] = 'L\'Organization n\'a pas de description (recommandé pour le GEO).';
        }
        
        if (empty($org['logo'])) {
            $results['warnings'][] = 'L\'Organization n\'a pas de logo (recommandé pour apparaître dans les résultats IA).';
        }
        
        $results['info'][] = 'Organization valide : "' . ($org['name'] ?? 'Sans nom') . '"';
    }

    // --- B. Vérifier les Person ---
    $persons = array_filter($entities, function ($entity) {
        return ($entity['@type'] ?? '') === 'Person';
    });

    $person_count = count($persons);
    $results['info'][] = sprintf('Entités Person trouvées : %d', $person_count);

    foreach ($persons as $person) {
        $person_name = $person['name'] ?? 'Personne sans nom';
        
        if (empty($person['@id'])) {
            $results['errors'][] = sprintf('La personne "%s" n\'a pas de @id.', $person_name);
        }

        if (empty($person['name'])) {
            $results['errors'][] = 'Une entité Person n\'a pas de nom.';
        }

        if (empty($person['url'])) {
            $results['warnings'][] = sprintf('La personne "%s" n\'a pas d\'URL.', $person_name);
        }

        if (empty($person['worksFor'])) {
            $results['warnings'][] = sprintf(
                'La personne "%s" n\'est pas reliée à une Organization (worksFor manquant). Cela affaiblit le signal d\'autorité.',
                $person_name
            );
        }
    }

    // --- C. Vérifications génériques pour toutes les entités ---
    foreach ($entities as $entity) {
        $entity_type = $entity['@type'] ?? 'Type inconnu';
        $entity_name = $entity['name'] ?? 'Sans nom';
        
        if (empty($entity['@id'])) {
            $results['errors'][] = sprintf(
                'Une entité de type "%s" est dépourvue de @id.',
                $entity_type
            );
        }

        if (empty($entity['name'])) {
            $results['warnings'][] = sprintf(
                'Une entité de type "%s" n\'a pas de nom.',
                $entity_type
            );
        }
        
        // Vérifier que le @id est bien formé
        if (!empty($entity['@id'])) {
            $id = $entity['@id'];
            $home_url = home_url();
            
            if (strpos($id, $home_url) !== 0) {
                $results['warnings'][] = sprintf(
                    'Le @id de "%s" ne commence pas par l\'URL du site (%s). Cela peut créer des ambiguïtés.',
                    $entity_name,
                    $home_url
                );
            }
            
            if (strpos($id, '#') === false) {
                $results['warnings'][] = sprintf(
                    'Le @id de "%s" ne contient pas de fragment (#). Format recommandé : %s#type-slug',
                    $entity_name,
                    $home_url
                );
            }
        }
    }

    // --- D. Vérifier les doublons de @id ---
    $ids = array_column($entities, '@id');
    $duplicates = array_filter(array_count_values($ids), function($count) {
        return $count > 1;
    });
    
    if (!empty($duplicates)) {
        foreach ($duplicates as $id => $count) {
            $results['errors'][] = sprintf(
                'Le @id "%s" est utilisé %d fois. Chaque entité doit avoir un identifiant unique.',
                $id,
                $count
            );
        }
    }

    // --- E. Message de synthèse ---
    if (empty($results['errors']) && empty($results['warnings'])) {
        $results['info'][] = '✅ Aucune incohérence détectée. Le graphe d\'entités est propre et optimisé pour le GEO.';
    } else {
        $error_count = count($results['errors']);
        $warning_count = count($results['warnings']);
        
        if ($error_count > 0) {
            $results['info'][] = sprintf(
                '❌ %d erreur(s) critique(s) à corriger.',
                $error_count
            );
        }
        
        if ($warning_count > 0) {
            $results['info'][] = sprintf(
                '⚠️ %d avertissement(s) à considérer pour améliorer votre visibilité GEO.',
                $warning_count
            );
        }
    }

    return $results;
}

/**
 * Récupère les statistiques rapides sur les entités
 */
function geo_get_entity_stats(): array {
    $entities = geo_get_entities();
    
    $stats = [
        'total' => count($entities),
        'types' => []
    ];
    
    foreach ($entities as $entity) {
        $type = $entity['@type'] ?? 'Unknown';
        $stats['types'][$type] = ($stats['types'][$type] ?? 0) + 1;
    }
    
    return $stats;
}