<?php
/**
 * Plugin Name: GEO Entities + EAS Bridge
 * Description: Fait le pont entre GEO Entities Core et Entity Authority Signals
 * Version: 1.0.3
 * Author: Erwan Tanguy - Ticoët
 * Requires Plugins: geo-entities, entity-authority-signals
 * 
 * V1.0.3 - CORRECTION CRITIQUE :
 * - Filtre pour supprimer @context des entités individuelles
 * - Validation stricte du @type (ne doit JAMAIS être "worksFor")
 * - Nettoyage des entités mal formées
 */

if (!defined('ABSPATH')) {
    exit;
}

define('GEO_EAS_BRIDGE_VERSION', '1.0.3');

/**
 * Enregistre les entités EAS dans le registre GEO Entities
 */
add_action('wp_head', function () {
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('=== GEO EAS Bridge: Début enregistrement ===');
    }
    
    $eas_entities = get_posts([
        'post_type' => 'entity',
        'posts_per_page' => -1,
        'post_status' => 'publish',
    ]);
    
    if (empty($eas_entities)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('GEO EAS Bridge: Aucune entité EAS trouvée');
        }
        return;
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log(sprintf('GEO EAS Bridge: %d entité(s) EAS à traiter', count($eas_entities)));
    }
    
    $registered_count = 0;
    
    foreach ($eas_entities as $entity_post) {
        $entity_data = eas_bridge_build_entity($entity_post);
        
        if ($entity_data) {
            // ✅ VALIDATION : Ne jamais enregistrer une entité avec @type invalide
            if (isset($entity_data['@type']) && in_array($entity_data['@type'], ['worksFor', 'memberOf'])) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log(sprintf(
                        '❌ GEO EAS Bridge: ENTITÉ REJETÉE - @type invalide "%s" pour "%s"',
                        $entity_data['@type'],
                        $entity_data['name'] ?? 'Sans nom'
                    ));
                }
                continue; // Passer à l'entité suivante
            }
            
            // ✅ NETTOYAGE : Supprimer @context si présent (sera ajouté globalement)
            if (isset($entity_data['@context'])) {
                unset($entity_data['@context']);
            }
            
            // Enregistrer dans GEO Entities
            geo_register_entity($entity_data);
            $registered_count++;
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log(sprintf(
                    '✅ GEO EAS Bridge: Entité enregistrée - %s (%s) - @id: %s',
                    $entity_data['name'] ?? 'Sans nom',
                    $entity_data['@type'] ?? 'Sans type',
                    $entity_data['@id'] ?? 'Sans @id'
                ));
            }
        }
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log(sprintf('GEO EAS Bridge: %d entité(s) enregistrée(s) dans GEO', $registered_count));
        error_log('=== GEO EAS Bridge: Fin enregistrement ===');
    }
    
}, 18);

/**
 * Génère un @id sémantique pour une entité
 */
function eas_bridge_generate_semantic_id($post_id, $type, $name) {
    
    // ✅ VALIDATION : Ne jamais utiliser "worksFor" ou "memberOf" comme type
    if (in_array(strtolower($type), ['worksfor', 'memberof'])) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('❌ Type invalide détecté : ' . $type);
        }
        return home_url('/#entity-' . $post_id); // Fallback
    }
    
    $type_slug = strtolower($type);
    $name_slug = sanitize_title($name);
    
    // Limiter la longueur
    if (strlen($name_slug) > 50) {
        $name_slug = substr($name_slug, 0, 50);
    }
    
    // Fallback si le slug est vide
    if ($name_slug === '' || $name_slug === '-') {
        return home_url('/#entity-' . $post_id);
    }
    
    return home_url('/#' . $type_slug . '-' . $name_slug);
}

/**
 * Construit une entité au format GEO Entities
 */
function eas_bridge_build_entity($post) {
    
    $post_id = $post->ID;
    
    // Récupérer le type
    $types = wp_get_post_terms($post_id, 'entity_type');
    if (!$types || is_wp_error($types)) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf('❌ Pas de type pour entité #%d ("%s")', $post_id, get_the_title($post)));
        }
        return null;
    }
    
    $type = $types[0]->name;
    
    // ✅ VALIDATION CRITIQUE : Le type ne doit JAMAIS être "worksFor" ou "memberOf"
    if (in_array(strtolower($type), ['worksfor', 'memberof'])) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf(
                '❌ TYPE INVALIDE DÉTECTÉ pour entité #%d: "%s" - Cette entité sera IGNORÉE',
                $post_id,
                $type
            ));
        }
        return null; // Ne pas construire cette entité
    }
    
    // Données de base
    $canonical = get_post_meta($post_id, '_entity_canonical', true);
    $name = !empty($canonical) ? $canonical : get_the_title($post);
    $description = wp_strip_all_tags($post->post_content);
    $url = get_post_meta($post_id, '_entity_url', true);
    
    // @id sémantique
    $id = eas_bridge_generate_semantic_id($post_id, $type, $name);
    
    // ✅ Structure de base SANS @context (GEO l'ajoute globalement)
    $entity = [
        '@type' => $type,
        '@id' => $id,
        'name' => $name,
    ];
    
    if (!empty($description)) {
        $entity['description'] = $description;
    }
    
    if (!empty($url)) {
        $entity['url'] = $url;
    }
    
    // Image / Logo
    $image = get_post_meta($post_id, '_entity_image', true);
    if (empty($image) && has_post_thumbnail($post_id)) {
        $image = get_the_post_thumbnail_url($post_id, 'full');
    }
    if (!empty($image)) {
        $entity['image'] = $image;
        if ($type === 'Organization') {
            $entity['logo'] = $image;
        }
    }
    
    // Synonymes
    $synonyms = get_post_meta($post_id, '_entity_synonyms', true);
    if (!empty($synonyms)) {
        $synonyms_array = array_map('trim', explode(',', $synonyms));
        $synonyms_array = array_filter($synonyms_array);
        
        if (!empty($synonyms_array)) {
            if (count($synonyms_array) === 1) {
                $entity['alternateName'] = $synonyms_array[0];
            } else {
                $entity['alternateName'] = $synonyms_array;
            }
        }
    }
    
    // SameAs
    $same_as = get_post_meta($post_id, '_entity_same_as', true);
    if (!empty($same_as)) {
        $same_as_array = array_filter(array_map('trim', explode("\n", $same_as)));
        if (!empty($same_as_array)) {
            $entity['sameAs'] = $same_as_array;
        }
    }
    
    // Propriétés spécifiques
    switch ($type) {
        case 'Person':
            $entity = eas_bridge_add_person_properties($entity, $post_id);
            break;
            
        case 'Organization':
        case 'LocalBusiness':
            $entity = eas_bridge_add_organization_properties($entity, $post_id);
            break;
    }
    
    return $entity;
}

/**
 * Ajoute les propriétés Person
 */
function eas_bridge_add_person_properties($entity, $post_id) {
    
    $job_title = get_post_meta($post_id, '_entity_job_title', true);
    if (!empty($job_title)) {
        $entity['jobTitle'] = $job_title;
    }
    
    $email = get_post_meta($post_id, '_entity_email', true);
    if (!empty($email)) {
        $entity['email'] = $email;
    }
    
    $telephone = get_post_meta($post_id, '_entity_telephone', true);
    if (!empty($telephone)) {
        $entity['telephone'] = $telephone;
    }
    
    // WorksFor - Référence simple
    $works_for = get_post_meta($post_id, '_entity_works_for', true);
    if (!empty($works_for)) {
        if ($works_for === 'geo_entities_org') {
            $entity['worksFor'] = [
                '@id' => home_url('/#organization'),
            ];
        } else {
            // Récupérer l'Organization référencée
            $org_post = get_post($works_for);
            if ($org_post && $org_post->post_type === 'entity') {
                $org_name = get_the_title($org_post);
                $org_id = eas_bridge_generate_semantic_id($works_for, 'Organization', $org_name);
                
                $entity['worksFor'] = [
                    '@id' => $org_id,
                ];
            }
        }
    }
    
    // MemberOf
    $member_of = get_post_meta($post_id, '_entity_member_of', true);
    if (!empty($member_of)) {
        if ($member_of === 'geo_entities_org') {
            $entity['memberOf'] = [
                '@id' => home_url('/#organization'),
            ];
        } else {
            $org_post = get_post($member_of);
            if ($org_post && $org_post->post_type === 'entity') {
                $org_name = get_the_title($org_post);
                $org_id = eas_bridge_generate_semantic_id($member_of, 'Organization', $org_name);
                
                $entity['memberOf'] = [
                    '@id' => $org_id,
                ];
            }
        }
    }
    
    return $entity;
}

/**
 * Ajoute les propriétés Organization
 */
function eas_bridge_add_organization_properties($entity, $post_id) {
    
    $email = get_post_meta($post_id, '_entity_email', true);
    if (!empty($email)) {
        $entity['email'] = $email;
    }
    
    $telephone = get_post_meta($post_id, '_entity_telephone', true);
    if (!empty($telephone)) {
        $entity['telephone'] = $telephone;
    }
    
    // Adresse
    $street = get_post_meta($post_id, '_entity_address_street', true);
    $city = get_post_meta($post_id, '_entity_address_city', true);
    $postal = get_post_meta($post_id, '_entity_address_postal', true);
    $country = get_post_meta($post_id, '_entity_address_country', true);
    
    if (!empty($street) || !empty($city)) {
        $address = [
            '@type' => 'PostalAddress',
        ];
        
        if (!empty($street)) $address['streetAddress'] = $street;
        if (!empty($city)) $address['addressLocality'] = $city;
        if (!empty($postal)) $address['postalCode'] = $postal;
        if (!empty($country)) $address['addressCountry'] = $country;
        
        $entity['address'] = $address;
    }
    
    return $entity;
}

/**
 * ✅ NOUVEAU : Filtre pour nettoyer les entités avant affichage final
 * Supprime @context des entités individuelles et valide les @type
 */
add_filter('geo_entities_before_output', function($entities) {
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log('=== Nettoyage final des entités avant affichage ===');
    }
    
    $cleaned = [];
    
    foreach ($entities as $entity) {
        // Supprimer @context si présent
        if (isset($entity['@context'])) {
            unset($entity['@context']);
        }
        
        // Vérifier que @type est valide
        if (!isset($entity['@type'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('❌ Entité sans @type ignorée');
            }
            continue;
        }
        
        // Rejeter les @type invalides
        if (in_array($entity['@type'], ['worksFor', 'memberOf'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log(sprintf(
                    '❌ Entité avec @type invalide "%s" rejetée: %s',
                    $entity['@type'],
                    $entity['name'] ?? 'Sans nom'
                ));
            }
            continue;
        }
        
        $cleaned[] = $entity;
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_log(sprintf('✅ %d entités nettoyées sur %d initiales', count($cleaned), count($entities)));
    }
    
    return $cleaned;
    
}, 10);

/**
 * Notice admin
 */
add_action('admin_notices', function () {
    
    $screen = get_current_screen();
    
    if ($screen && (
        $screen->post_type === 'entity' || 
        $screen->id === 'toplevel_page_geo-entity-audit'
    )) {
        
        $eas_count = wp_count_posts('entity');
        $eas_published = $eas_count->publish ?? 0;
        
        if ($eas_published > 0) {
            ?>
            <div class="notice notice-info">
                <p>
                    <strong>🌉 GEO EAS Bridge V<?php echo GEO_EAS_BRIDGE_VERSION; ?> actif</strong><br>
                    <?php echo $eas_published; ?> entité(s) d'Entity Authority Signals intégrées.
                </p>
                <?php if ($screen->post_type === 'entity'): ?>
                <p>
                    <a href="<?php echo admin_url('admin.php?page=geo-entity-audit'); ?>" class="button button-small">
                        📊 Voir l'audit GEO
                    </a>
                </p>
                <?php endif; ?>
            </div>
            <?php
        }
    }
});

/**
 * Colonne dans la liste
 */
add_filter('manage_entity_posts_columns', function ($columns) {
    $columns['geo_bridge'] = '🌉 GEO Bridge';
    return $columns;
}, 20);

add_action('manage_entity_posts_custom_column', function ($column, $post_id) {
    if ($column === 'geo_bridge') {
        $types = wp_get_post_terms($post_id, 'entity_type');
        $type = $types && !is_wp_error($types) ? $types[0]->name : 'Thing';
        $name = get_the_title($post_id);
        $semantic_id = eas_bridge_generate_semantic_id($post_id, $type, $name);
        
        $fragment = parse_url($semantic_id, PHP_URL_FRAGMENT);
        
        echo '<span style="color: green; display: block;">✅ Intégré</span>';
        echo '<code style="font-size: 11px; color: #666;">#' . esc_html($fragment) . '</code>';
    }
}, 20, 2);

/**
 * Meta box
 */
add_action('add_meta_boxes', function () {
    add_meta_box(
        'eas_geo_bridge_status',
        '🌉 Statut GEO Bridge',
        'eas_geo_bridge_status_meta_box',
        'entity',
        'side',
        'high'
    );
}, 25);

function eas_geo_bridge_status_meta_box($post) {
    
    $types = wp_get_post_terms($post->ID, 'entity_type');
    $type = $types && !is_wp_error($types) ? $types[0]->name : 'Thing';
    $name = get_the_title($post);
    $semantic_id = eas_bridge_generate_semantic_id($post->ID, $type, $name);
    
    ?>
    <div style="padding: 10px;">
        <p style="margin: 0 0 10px;">
            <strong style="color: green;">✅ Intégré à GEO Entities</strong>
        </p>
        
        <p style="margin: 10px 0; padding: 10px; background: #f0f0f0; border-radius: 4px;">
            <strong>@id sémantique :</strong><br>
            <code style="font-size: 11px; word-break: break-all;">
                <?php echo esc_html($semantic_id); ?>
            </code>
        </p>
        
        <p style="margin: 0; font-size: 12px; color: #666;">
            Cette entité est automatiquement enregistrée dans GEO Entities avec un @id lisible.
        </p>
        
        <p style="margin: 10px 0 0;">
            <a href="<?php echo admin_url('admin.php?page=geo-entity-audit'); ?>" 
               class="button button-small button-primary" 
               style="width: 100%;">
                📊 Voir l'audit GEO
            </a>
        </p>
    </div>
    <?php
}