<?php
/**
 * Plugin Name: GEO Content Audit
 * Description: Audit du contenu GEO - Détecte FAQ, blockquotes, images, sons, vidéos et entités
 * Version: 1.1.0
 * Author: Erwan Tanguy - Ticoët
 * Requires Plugins: geo-entities
 * 
 * NOTE: Ce plugin ne génère PAS de Schema.org, il AUDITE uniquement
 * vos plugins existants qui génèrent déjà le Schema.org.
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Ajoute la page d'audit
 */
add_action('admin_menu', function () {
    // Vérifier que le menu parent existe
    global $menu;
    
    $parent_exists = false;
    foreach ($menu as $item) {
        if (isset($item[2]) && $item[2] === 'geo-entity-audit') {
            $parent_exists = true;
            break;
        }
    }
    
    if ($parent_exists) {
        add_submenu_page(
            'geo-entity-audit',
            'Audit du contenu',
            '🔍 Contenu',
            'manage_options',
            'geo-content-audit',
            'geo_render_content_audit_page'
        );
    } else {
        // Fallback : créer un menu indépendant
        add_menu_page(
            'GEO Content Audit',
            'GEO Contenu',
            'manage_options',
            'geo-content-audit',
            'geo_render_content_audit_page',
            'dashicons-search',
            82
        );
    }
}, 15);

/**
 * Page d'audit du contenu
 */
function geo_render_content_audit_page() {
    
    // Récupérer les articles
    $posts = get_posts([
        'post_type' => 'post',
        'posts_per_page' => 50,
        'post_status' => 'publish',
        'orderby' => 'date',
        'order' => 'DESC',
    ]);
    
    $audit_results = [];
    $stats = [
        'total' => count($posts),
        'with_faq' => 0,
        'with_blockquotes' => 0,
        'with_images' => 0,
        'with_audio' => 0,
        'with_video' => 0,
        'with_entities' => 0,
        'score_excellent' => 0,
        'score_good' => 0,
        'score_poor' => 0,
    ];
    
    foreach ($posts as $post) {
        $audit = geo_audit_content($post);
        $audit_results[] = $audit;
        
        if ($audit['has_faq']) $stats['with_faq']++;
        if ($audit['has_blockquotes']) $stats['with_blockquotes']++;
        if ($audit['has_images']) $stats['with_images']++;
        if ($audit['has_audio']) $stats['with_audio']++;
        if ($audit['has_video']) $stats['with_video']++;
        if ($audit['has_entities']) $stats['with_entities']++;
        
        if ($audit['geo_score'] >= 80) $stats['score_excellent']++;
        elseif ($audit['geo_score'] >= 50) $stats['score_good']++;
        else $stats['score_poor']++;
    }
    
    ?>
    <div class="wrap">
        <h1>📝 Audit GEO du contenu</h1>
        
        <p class="description">
            Audit de vos contenus pour optimiser leur visibilité dans les moteurs IA.
            Les Schema.org sont générés par vos plugins existants, cet audit vérifie leur présence.
        </p>
        
        <!-- Statistiques -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 15px; margin: 20px 0;">
            
            <div class="card" style="padding: 20px; text-align: center;">
                <h3 style="margin: 0; font-size: 32px; color: #0073aa;"><?php echo $stats['total']; ?></h3>
                <p style="margin: 5px 0 0; color: #666; font-size: 13px;">Articles</p>
            </div>
            
            <div class="card" style="padding: 20px; text-align: center;">
                <h3 style="margin: 0; font-size: 32px; color: #00a32a;"><?php echo $stats['score_excellent']; ?></h3>
                <p style="margin: 5px 0 0; color: #666; font-size: 13px;">Excellents (≥80)</p>
            </div>
            
            <div class="card" style="padding: 20px; text-align: center;">
                <h3 style="margin: 0; font-size: 32px; color: #dba617;"><?php echo $stats['score_good']; ?></h3>
                <p style="margin: 5px 0 0; color: #666; font-size: 13px;">Bons (≥50)</p>
            </div>
            
            <div class="card" style="padding: 20px; text-align: center;">
                <h3 style="margin: 0; font-size: 32px; color: #d63638;"><?php echo $stats['score_poor']; ?></h3>
                <p style="margin: 5px 0 0; color: #666; font-size: 13px;">À améliorer</p>
            </div>
            
        </div>
        
        <!-- Fonctionnalités GEO -->
        <div class="card">
            <h2>📊 Éléments GEO détectés</h2>
            <table class="wp-list-table widefat">
                <thead>
                    <tr>
                        <th>Type</th>
                        <th>Articles</th>
                        <th>%</th>
                        <th>Impact GEO</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td>❓ <strong>FAQ</strong></td>
                        <td><?php echo $stats['with_faq']; ?></td>
                        <td><?php echo round(($stats['with_faq'] / max($stats['total'], 1)) * 100); ?>%</td>
                        <td><span style="color: #00a32a;">★★★★★</span> Très élevé</td>
                    </tr>
                    <tr>
                        <td>💬 <strong>Citations (blockquote)</strong></td>
                        <td><?php echo $stats['with_blockquotes']; ?></td>
                        <td><?php echo round(($stats['with_blockquotes'] / max($stats['total'], 1)) * 100); ?>%</td>
                        <td><span style="color: #00a32a;">★★★★☆</span> Élevé</td>
                    </tr>
                    <tr>
                        <td>🖼️ <strong>Images</strong></td>
                        <td><?php echo $stats['with_images']; ?></td>
                        <td><?php echo round(($stats['with_images'] / max($stats['total'], 1)) * 100); ?>%</td>
                        <td><span style="color: #dba617;">★★★☆☆</span> Moyen</td>
                    </tr>
                    <tr>
                        <td>🎵 <strong>Audio</strong></td>
                        <td><?php echo $stats['with_audio']; ?></td>
                        <td><?php echo round(($stats['with_audio'] / max($stats['total'], 1)) * 100); ?>%</td>
                        <td><span style="color: #dba617;">★★☆☆☆</span> Faible</td>
                    </tr>
                    <tr>
                        <td>🎥 <strong>Vidéo</strong></td>
                        <td><?php echo $stats['with_video']; ?></td>
                        <td><?php echo round(($stats['with_video'] / max($stats['total'], 1)) * 100); ?>%</td>
                        <td><span style="color: #00a32a;">★★★★☆</span> Élevé</td>
                    </tr>
                    <tr>
                        <td>👥 <strong>Entités</strong></td>
                        <td><?php echo $stats['with_entities']; ?></td>
                        <td><?php echo round(($stats['with_entities'] / max($stats['total'], 1)) * 100); ?>%</td>
                        <td><span style="color: #00a32a;">★★★★★</span> Très élevé</td>
                    </tr>
                </tbody>
            </table>
        </div>
        
        <!-- Liste des articles -->
        <div class="card" style="margin-top: 20px;">
            <h2>📄 Détail des articles</h2>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 35%;">Article</th>
                        <th style="width: 12%;">Score</th>
                        <th style="width: 33%;">Éléments GEO</th>
                        <th style="width: 20%;">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($audit_results as $audit): ?>
                        <?php
                        $score_color = $audit['geo_score'] >= 80 ? '#00a32a' : ($audit['geo_score'] >= 50 ? '#dba617' : '#d63638');
                        $score_label = $audit['geo_score'] >= 80 ? 'Excellent' : ($audit['geo_score'] >= 50 ? 'Bon' : 'À améliorer');
                        ?>
                        <tr>
                            <td>
                                <strong>
                                    <a href="<?php echo get_permalink($audit['post_id']); ?>" target="_blank">
                                        <?php echo esc_html($audit['title']); ?>
                                    </a>
                                </strong>
                                <br>
                                <span style="font-size: 11px; color: #666;">
                                    <?php echo date('d/m/Y', strtotime($audit['date'])); ?>
                                </span>
                            </td>
                            
                            <td>
                                <div style="text-align: center;">
                                    <div style="width: 50px; height: 50px; margin: 0 auto; border-radius: 50%; background: <?php echo $score_color; ?>; display: flex; align-items: center; justify-content: center; color: white; font-weight: bold; font-size: 16px;">
                                        <?php echo $audit['geo_score']; ?>
                                    </div>
                                    <small style="color: <?php echo $score_color; ?>; font-weight: 600;">
                                        <?php echo $score_label; ?>
                                    </small>
                                </div>
                            </td>
                            
                            <td>
                                <div style="display: flex; flex-wrap: wrap; gap: 8px; font-size: 12px;">
                                    <span style="padding: 4px 8px; background: <?php echo $audit['has_faq'] ? '#d4edda' : '#f8d7da'; ?>; border-radius: 3px;">
                                        <?php echo $audit['has_faq'] ? '✅' : '❌'; ?> FAQ (<?php echo $audit['faq_count']; ?>)
                                    </span>
                                    <span style="padding: 4px 8px; background: <?php echo $audit['has_blockquotes'] ? '#d4edda' : '#f8d7da'; ?>; border-radius: 3px;">
                                        <?php echo $audit['has_blockquotes'] ? '✅' : '❌'; ?> Citations (<?php echo $audit['blockquote_count']; ?>)
                                    </span>
                                    <span style="padding: 4px 8px; background: <?php echo $audit['has_images'] ? '#d4edda' : '#f8d7da'; ?>; border-radius: 3px;">
                                        <?php echo $audit['has_images'] ? '✅' : '❌'; ?> Images (<?php echo $audit['image_count']; ?>)
                                    </span>
                                    <span style="padding: 4px 8px; background: <?php echo $audit['has_audio'] ? '#d4edda' : '#f8d7da'; ?>; border-radius: 3px;">
                                        <?php echo $audit['has_audio'] ? '✅' : '❌'; ?> Audio (<?php echo $audit['audio_count']; ?>)
                                    </span>
                                    <span style="padding: 4px 8px; background: <?php echo $audit['has_video'] ? '#d4edda' : '#f8d7da'; ?>; border-radius: 3px;">
                                        <?php echo $audit['has_video'] ? '✅' : '❌'; ?> Vidéo (<?php echo $audit['video_count']; ?>)
                                    </span>
                                    <span style="padding: 4px 8px; background: <?php echo $audit['has_entities'] ? '#d4edda' : '#f8d7da'; ?>; border-radius: 3px;">
                                        <?php echo $audit['has_entities'] ? '✅' : '❌'; ?> Entités (<?php echo $audit['entity_count']; ?>)
                                    </span>
                                </div>
                            </td>
                            
                            <td>
                                <a href="<?php echo get_edit_post_link($audit['post_id']); ?>" class="button button-small">
                                    ✏️ Modifier
                                </a>
                                <button type="button" class="button button-small" onclick="toggleDetails(<?php echo $audit['post_id']; ?>)">
                                    🔍 Détails
                                </button>
                            </td>
                        </tr>
                        
                        <!-- Détails (masqués) -->
                        <tr id="details-<?php echo $audit['post_id']; ?>" style="display: none;">
                            <td colspan="4" style="background: #f9f9f9; padding: 20px;">
                                <?php geo_render_audit_details($audit); ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Recommandations -->
        <div class="card" style="margin-top: 20px; background: #e7f3ff; border-left: 4px solid #2196f3;">
            <h2>💡 Comment améliorer votre score GEO</h2>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                <div>
                    <h3>Impact maximal (★★★★★)</h3>
                    <ul>
                        <li><strong>FAQ</strong> : Les IA citent directement vos réponses</li>
                        <li><strong>Entités</strong> : Renforce votre autorité d'expert</li>
                    </ul>
                </div>
                <div>
                    <h3>Impact important (★★★★☆)</h3>
                    <ul>
                        <li><strong>Citations</strong> : Crédibilité et backlinks</li>
                        <li><strong>Vidéos</strong> : Engagement et multimédia</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    function toggleDetails(postId) {
        const row = document.getElementById('details-' + postId);
        row.style.display = row.style.display === 'none' ? 'table-row' : 'none';
    }
    </script>
    <?php
}

/**
 * Audit d'un article
 */
function geo_audit_content($post) {
    
    $content = $post->post_content;
    
    $audit = [
        'post_id' => $post->ID,
        'title' => get_the_title($post),
        'date' => $post->post_date,
        'geo_score' => 0,
        
        'has_faq' => false,
        'faq_count' => 0,
        'faq_type' => '', // Nouveau : type de FAQ détecté
        
        'has_blockquotes' => false,
        'blockquote_count' => 0,
        'blockquotes_with_cite' => 0,
        
        'has_images' => false,
        'image_count' => 0,
        'images_with_alt' => 0,
        
        'has_audio' => false,
        'audio_count' => 0,
        
        'has_video' => false,
        'video_count' => 0,
        
        'has_entities' => false,
        'entity_count' => 0,
        
        'recommendations' => [],
    ];
    
    // ✅ FAQ - Détection de 3 formats différents
    
    // Format 1 : Bloc FAQGEO (details/summary)
    preg_match_all('/<details[^>]*>.*?<summary[^>]*>.*?<\/summary>.*?<\/details>/is', $content, $faq_details);
    $faq_details_count = count($faq_details[0]);
    
    // Format 2 : H3 + paragraphe classique
    preg_match_all('/<h3[^>]*>.*?<\/h3>\s*<p>.*?<\/p>/is', $content, $faq_h3);
    $faq_h3_count = count($faq_h3[0]);
    
    // Format 3 : JSON-LD FAQPage dans le contenu
    $has_faq_schema = preg_match('/"@type"\s*:\s*"FAQPage"/i', $content);
    
    // Compter le total et déterminer le type
    $audit['faq_count'] = max($faq_details_count, $faq_h3_count);
    
    if ($faq_details_count > 0) {
        $audit['faq_type'] = 'FAQGEO (details/summary)';
        $audit['has_faq'] = true;
    } elseif ($faq_h3_count >= 2) {
        $audit['faq_type'] = 'H3 + paragraphe';
        $audit['has_faq'] = true;
    } elseif ($has_faq_schema) {
        $audit['faq_type'] = 'FAQPage Schema.org détecté';
        $audit['has_faq'] = true;
    }
    
    if ($audit['has_faq']) {
        $audit['geo_score'] += 30;
    } else {
        $audit['recommendations'][] = 'Ajouter des FAQ (min 2) avec le bloc FAQGEO ou format H3+paragraphe';
    }
    
    // Blockquotes
    preg_match_all('/<blockquote[^>]*>.*?<\/blockquote>/is', $content, $quote_matches);
    $audit['blockquote_count'] = count($quote_matches[0]);
    $audit['has_blockquotes'] = $audit['blockquote_count'] > 0;
    
    foreach ($quote_matches[0] as $quote) {
        if (preg_match('/cite=["\'][^"\']+["\']/i', $quote)) {
            $audit['blockquotes_with_cite']++;
        }
    }
    
    if ($audit['has_blockquotes']) {
        $audit['geo_score'] += 15;
        if ($audit['blockquotes_with_cite'] > 0) {
            $audit['geo_score'] += 5;
        }
    } else {
        $audit['recommendations'][] = 'Ajouter des citations (blockquote avec cite)';
    }
    
    // Images
    preg_match_all('/<img[^>]+>/i', $content, $image_matches);
    $audit['image_count'] = count($image_matches[0]);
    $audit['has_images'] = $audit['image_count'] > 0;
    
    foreach ($image_matches[0] as $img) {
        if (preg_match('/alt=["\'][^"\']+["\']/i', $img)) {
            $audit['images_with_alt']++;
        }
    }
    
    if ($audit['has_images']) {
        $audit['geo_score'] += min(15, $audit['images_with_alt'] * 3);
    } else {
        $audit['recommendations'][] = 'Ajouter des images avec alt text';
    }
    
    // Audio
    preg_match_all('/<audio[^>]*>|wp:audio/', $content, $audio_matches);
    $audit['audio_count'] = count($audio_matches[0]);
    $audit['has_audio'] = $audit['audio_count'] > 0;
    if ($audit['has_audio']) {
        $audit['geo_score'] += 5;
    }
    
    // Vidéo
    preg_match_all('/<video[^>]*>|wp:video|wp:embed/', $content, $video_matches);
    $audit['video_count'] = count($video_matches[0]);
    $audit['has_video'] = $audit['video_count'] > 0;
    if ($audit['has_video']) {
        $audit['geo_score'] += 10;
    }
    
    // Entités
    preg_match_all('/\[entity id=\d+\]/', $content, $entity_matches);
    $audit['entity_count'] = count($entity_matches[0]);
    $audit['has_entities'] = $audit['entity_count'] > 0;
    if ($audit['has_entities']) {
        $audit['geo_score'] += 20;
    } else {
        $audit['recommendations'][] = 'Mentionner des entités avec [entity id=X]';
    }
    
    // Image à la une
    if (has_post_thumbnail($post)) {
        $audit['geo_score'] += 5;
    }
    
    // Limiter à 100
    $audit['geo_score'] = min(100, $audit['geo_score']);
    
    return $audit;
}

/**
 * Affiche les détails d'un audit
 */
function geo_render_audit_details($audit) {
    ?>
    <h3 style="margin-top: 0;">📊 Analyse détaillée</h3>
    
    <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 15px; margin-bottom: 20px;">
        
        <div style="padding: 15px; background: white; border-radius: 4px; border-left: 3px solid #2196f3;">
            <h4 style="margin: 0 0 10px;">❓ FAQ</h4>
            <p style="margin: 0; font-size: 24px; font-weight: bold; color: <?php echo $audit['has_faq'] ? '#00a32a' : '#d63638'; ?>;">
                <?php echo $audit['faq_count']; ?>
            </p>
            <p style="margin: 5px 0 0; font-size: 12px; color: #666;">
                <?php echo $audit['has_faq'] ? '✅ FAQPage Schema généré' : '❌ Min 2 pour FAQPage'; ?>
            </p>
        </div>
        
        <div style="padding: 15px; background: white; border-radius: 4px; border-left: 3px solid #9c27b0;">
            <h4 style="margin: 0 0 10px;">💬 Citations</h4>
            <p style="margin: 0; font-size: 24px; font-weight: bold; color: <?php echo $audit['has_blockquotes'] ? '#00a32a' : '#d63638'; ?>;">
                <?php echo $audit['blockquote_count']; ?>
            </p>
            <p style="margin: 5px 0 0; font-size: 12px; color: #666;">
                <?php echo $audit['blockquotes_with_cite']; ?> avec source (cite)
            </p>
        </div>
        
        <div style="padding: 15px; background: white; border-radius: 4px; border-left: 3px solid #4caf50;">
            <h4 style="margin: 0 0 10px;">🖼️ Images</h4>
            <p style="margin: 0; font-size: 24px; font-weight: bold; color: <?php echo $audit['has_images'] ? '#00a32a' : '#d63638'; ?>;">
                <?php echo $audit['image_count']; ?>
            </p>
            <p style="margin: 5px 0 0; font-size: 12px; color: #666;">
                <?php echo $audit['images_with_alt']; ?> avec alt text
            </p>
        </div>
        
        <div style="padding: 15px; background: white; border-radius: 4px; border-left: 3px solid #ff9800;">
            <h4 style="margin: 0 0 10px;">🎵 Audio</h4>
            <p style="margin: 0; font-size: 24px; font-weight: bold;">
                <?php echo $audit['audio_count']; ?>
            </p>
        </div>
        
        <div style="padding: 15px; background: white; border-radius: 4px; border-left: 3px solid #f44336;">
            <h4 style="margin: 0 0 10px;">🎥 Vidéo</h4>
            <p style="margin: 0; font-size: 24px; font-weight: bold;">
                <?php echo $audit['video_count']; ?>
            </p>
        </div>
        
        <div style="padding: 15px; background: white; border-radius: 4px; border-left: 3px solid #00bcd4;">
            <h4 style="margin: 0 0 10px;">👥 Entités</h4>
            <p style="margin: 0; font-size: 24px; font-weight: bold; color: <?php echo $audit['has_entities'] ? '#00a32a' : '#d63638'; ?>;">
                <?php echo $audit['entity_count']; ?>
            </p>
        </div>
        
    </div>
    
    <?php if (!empty($audit['recommendations'])): ?>
        <div style="padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107; border-radius: 4px;">
            <h4 style="margin: 0 0 10px;">💡 Recommandations</h4>
            <ul style="margin: 0;">
                <?php foreach ($audit['recommendations'] as $rec): ?>
                    <li><?php echo esc_html($rec); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>
    <?php
}

/**
 * Meta box dans l'éditeur
 */
add_action('add_meta_boxes', function () {
    add_meta_box(
        'geo_content_score',
        '📊 Score GEO',
        'geo_render_score_meta_box',
        'post',
        'side',
        'high'
    );
});

function geo_render_score_meta_box($post) {
    
    if ($post->post_status !== 'publish') {
        echo '<p style="color: #666; font-style: italic;">Publiez pour voir le score</p>';
        return;
    }
    
    $audit = geo_audit_content($post);
    $score_color = $audit['geo_score'] >= 80 ? '#00a32a' : ($audit['geo_score'] >= 50 ? '#dba617' : '#d63638');
    
    ?>
    <div style="text-align: center; padding: 15px;">
        <div style="width: 70px; height: 70px; margin: 0 auto; border-radius: 50%; background: <?php echo $score_color; ?>; display: flex; align-items: center; justify-content: center; color: white; font-weight: bold; font-size: 20px;">
            <?php echo $audit['geo_score']; ?>
        </div>
        <p style="margin: 10px 0; font-weight: 600;">
            <?php echo $audit['geo_score'] >= 80 ? 'Excellent' : ($audit['geo_score'] >= 50 ? 'Bon' : 'À améliorer'); ?>
        </p>
    </div>
    
    <div style="font-size: 13px;">
        <p style="margin: 5px 0;"><?php echo $audit['has_faq'] ? '✅' : '❌'; ?> FAQ (<?php echo $audit['faq_count']; ?>)</p>
        <p style="margin: 5px 0;"><?php echo $audit['has_blockquotes'] ? '✅' : '❌'; ?> Citations (<?php echo $audit['blockquote_count']; ?>)</p>
        <p style="margin: 5px 0;"><?php echo $audit['has_images'] ? '✅' : '❌'; ?> Images (<?php echo $audit['image_count']; ?>)</p>
        <p style="margin: 5px 0;"><?php echo $audit['has_audio'] ? '✅' : '❌'; ?> Audio (<?php echo $audit['audio_count']; ?>)</p>
        <p style="margin: 5px 0;"><?php echo $audit['has_video'] ? '✅' : '❌'; ?> Vidéo (<?php echo $audit['video_count']; ?>)</p>
        <p style="margin: 5px 0;"><?php echo $audit['has_entities'] ? '✅' : '❌'; ?> Entités (<?php echo $audit['entity_count']; ?>)</p>
    </div>
    
    <p style="margin-top: 15px;">
        <a href="<?php echo admin_url('admin.php?page=geo-content-audit'); ?>" 
           class="button button-primary button-small" 
           style="width: 100%;">
            📊 Audit complet
        </a>
    </p>
    <?php
}