<?php
/**
 * Meta boxes pour les détails des entités
 * V2.2 - Support de l'Organization GEO Entities
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Ajoute les meta boxes
 */
add_action('add_meta_boxes', function () {
    
    add_meta_box(
        'entity_details',
        'Détails de l\'entité',
        'eas_entity_details_meta_box',
        'entity',
        'normal',
        'high'
    );
    
    add_meta_box(
        'entity_schema_properties',
        'Propriétés Schema.org',
        'eas_entity_schema_meta_box',
        'entity',
        'normal',
        'high'
    );
    
    add_meta_box(
        'entity_relations',
        'Relations avec d\'autres entités',
        'eas_entity_relations_meta_box',
        'entity',
        'side',
        'default'
    );
    
});

/**
 * Meta box : Détails de base
 */
function eas_entity_details_meta_box($post) {
    wp_nonce_field('eas_entity_meta', 'eas_entity_nonce');
    
    $canonical = get_post_meta($post->ID, '_entity_canonical', true);
    $synonyms = get_post_meta($post->ID, '_entity_synonyms', true);
    $url = get_post_meta($post->ID, '_entity_url', true);
    ?>
    
    <table class="form-table">
        <tr>
            <th><label for="entity_canonical">Nom canonique</label></th>
            <td>
                <input type="text" 
                       id="entity_canonical" 
                       name="entity_canonical" 
                       value="<?php echo esc_attr($canonical); ?>" 
                       class="regular-text">
                <p class="description">Le nom officiel et unique de cette entité</p>
            </td>
        </tr>
        
        <tr>
            <th><label for="entity_url">URL officielle</label></th>
            <td>
                <input type="url" 
                       id="entity_url" 
                       name="entity_url" 
                       value="<?php echo esc_url($url); ?>" 
                       class="regular-text" 
                       placeholder="https://example.com">
                <p class="description">L'URL principale de cette entité (site web, page Wikipedia, etc.)</p>
            </td>
        </tr>
        
        <tr>
            <th><label for="entity_synonyms">Synonymes / Variantes</label></th>
            <td>
                <textarea id="entity_synonyms" 
                          name="entity_synonyms" 
                          rows="3" 
                          class="large-text"><?php echo esc_textarea($synonyms); ?></textarea>
                <p class="description">Variantes du nom, séparées par des virgules (ex: CEO, PDG, Directeur général)</p>
            </td>
        </tr>
    </table>
    
    <?php
}

/**
 * Meta box : Propriétés Schema.org spécifiques
 */
function eas_entity_schema_meta_box($post) {
    
    $types = wp_get_post_terms($post->ID, 'entity_type');
    $current_type = $types && !is_wp_error($types) ? $types[0]->name : '';
    
    $image = get_post_meta($post->ID, '_entity_image', true);
    $same_as = get_post_meta($post->ID, '_entity_same_as', true);
    
    $job_title = get_post_meta($post->ID, '_entity_job_title', true);
    $email = get_post_meta($post->ID, '_entity_email', true);
    $telephone = get_post_meta($post->ID, '_entity_telephone', true);
    
    $address_street = get_post_meta($post->ID, '_entity_address_street', true);
    $address_city = get_post_meta($post->ID, '_entity_address_city', true);
    $address_postal = get_post_meta($post->ID, '_entity_address_postal', true);
    $address_country = get_post_meta($post->ID, '_entity_address_country', true);
    
    ?>
    
    <div class="eas-schema-properties">
        
        <h4>Propriétés communes</h4>
        <table class="form-table">
            <tr>
                <th><label for="entity_image">Image / Logo</label></th>
                <td>
                    <input type="url" 
                           id="entity_image" 
                           name="entity_image" 
                           value="<?php echo esc_url($image); ?>" 
                           class="regular-text" 
                           placeholder="https://example.com/image.jpg">
                    <p class="description">URL de l'image ou logo (ou utilisez l'image à la une)</p>
                </td>
            </tr>
            
            <tr>
                <th><label for="entity_same_as">Liens sameAs</label></th>
                <td>
                    <textarea id="entity_same_as" 
                              name="entity_same_as" 
                              rows="4" 
                              class="large-text" 
                              placeholder="https://facebook.com/...&#10;https://twitter.com/...&#10;https://linkedin.com/..."><?php echo esc_textarea($same_as); ?></textarea>
                    <p class="description">Liens vers les profils sociaux ou pages externes (un par ligne)</p>
                </td>
            </tr>
        </table>
        
        <?php if ($current_type === 'Person'): ?>
        <h4>Propriétés Person</h4>
        <table class="form-table">
            <tr>
                <th><label for="entity_job_title">Fonction / Titre</label></th>
                <td>
                    <input type="text" 
                           id="entity_job_title" 
                           name="entity_job_title" 
                           value="<?php echo esc_attr($job_title); ?>" 
                           class="regular-text" 
                           placeholder="CEO, Développeur, etc.">
                </td>
            </tr>
            
            <tr>
                <th><label for="entity_email">Email</label></th>
                <td>
                    <input type="email" 
                           id="entity_email" 
                           name="entity_email" 
                           value="<?php echo esc_attr($email); ?>" 
                           class="regular-text">
                </td>
            </tr>
            
            <tr>
                <th><label for="entity_telephone">Téléphone</label></th>
                <td>
                    <input type="tel" 
                           id="entity_telephone" 
                           name="entity_telephone" 
                           value="<?php echo esc_attr($telephone); ?>" 
                           class="regular-text">
                </td>
            </tr>
        </table>
        <?php endif; ?>
        
        <?php if (in_array($current_type, ['Organization', 'LocalBusiness'])): ?>
        <h4>Adresse postale</h4>
        <table class="form-table">
            <tr>
                <th><label for="entity_address_street">Rue</label></th>
                <td>
                    <input type="text" 
                           id="entity_address_street" 
                           name="entity_address_street" 
                           value="<?php echo esc_attr($address_street); ?>" 
                           class="regular-text">
                </td>
            </tr>
            
            <tr>
                <th><label for="entity_address_city">Ville</label></th>
                <td>
                    <input type="text" 
                           id="entity_address_city" 
                           name="entity_address_city" 
                           value="<?php echo esc_attr($address_city); ?>" 
                           class="regular-text">
                </td>
            </tr>
            
            <tr>
                <th><label for="entity_address_postal">Code postal</label></th>
                <td>
                    <input type="text" 
                           id="entity_address_postal" 
                           name="entity_address_postal" 
                           value="<?php echo esc_attr($address_postal); ?>" 
                           class="regular-text">
                </td>
            </tr>
            
            <tr>
                <th><label for="entity_address_country">Pays</label></th>
                <td>
                    <input type="text" 
                           id="entity_address_country" 
                           name="entity_address_country" 
                           value="<?php echo esc_attr($address_country); ?>" 
                           class="regular-text" 
                           placeholder="FR">
                    <p class="description">Code pays ISO (FR, US, GB, etc.)</p>
                </td>
            </tr>
        </table>
        <?php endif; ?>
        
    </div>
    
    <?php
}

/**
 * Meta box : Relations entre entités
 */
function eas_entity_relations_meta_box($post) {
    
    $works_for = get_post_meta($post->ID, '_entity_works_for', true);
    $member_of = get_post_meta($post->ID, '_entity_member_of', true);
    
    // Récupérer toutes les entités Organization
    $organizations = get_posts([
        'post_type' => 'entity',
        'posts_per_page' => -1,
        'tax_query' => [
            [
                'taxonomy' => 'entity_type',
                'field' => 'name',
                'terms' => 'Organization',
            ],
        ],
    ]);
    
    // Vérifier si GEO Entities est actif et génère une Organization
    $geo_org_available = false;
    if (function_exists('get_bloginfo')) {
        $site_name = get_bloginfo('name');
        if (!empty($site_name)) {
            $geo_org_available = true;
        }
    }
    
    ?>
    
    <p>
        <label for="entity_works_for"><strong>Travaille pour (worksFor)</strong></label><br>
        <select id="entity_works_for" name="entity_works_for" style="width: 100%;">
            <option value="">-- Aucune --</option>
            
            <?php if ($geo_org_available): ?>
                <option value="geo_entities_org" <?php selected($works_for, 'geo_entities_org'); ?>>
                    🌍 <?php echo esc_html(get_bloginfo('name')); ?> (GEO Entities - principale)
                </option>
            <?php endif; ?>
            
            <?php foreach ($organizations as $org): ?>
                <option value="<?php echo $org->ID; ?>" <?php selected($works_for, $org->ID); ?>>
                    <?php echo esc_html($org->post_title); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <span class="description">Pour les Person : l'organisation employeur</span>
    </p>
    
    <p>
        <label for="entity_member_of"><strong>Membre de (memberOf)</strong></label><br>
        <select id="entity_member_of" name="entity_member_of" style="width: 100%;">
            <option value="">-- Aucune --</option>
            
            <?php if ($geo_org_available): ?>
                <option value="geo_entities_org" <?php selected($member_of, 'geo_entities_org'); ?>>
                    🌍 <?php echo esc_html(get_bloginfo('name')); ?> (GEO Entities - principale)
                </option>
            <?php endif; ?>
            
            <?php foreach ($organizations as $org): ?>
                <option value="<?php echo $org->ID; ?>" <?php selected($member_of, $org->ID); ?>>
                    <?php echo esc_html($org->post_title); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <span class="description">Pour les Person : organisation dont la personne est membre</span>
    </p>
    
    <?php if ($geo_org_available): ?>
    <div style="margin-top: 15px; padding: 10px; background: #e7f3ff; border-left: 3px solid #0073aa; border-radius: 4px;">
        <p style="margin: 0; font-size: 12px; color: #004085;">
            <strong>💡 Astuce :</strong> L'option "GEO Entities (principale)" référence l'Organization générée automatiquement par le plugin GEO Entities. Utilisez-la pour éviter de créer un doublon.
        </p>
    </div>
    <?php endif; ?>
    
    <script>
    // Indiquer que GEO Entities Organization est disponible
    window.geoEntitiesOrgExists = <?php echo $geo_org_available ? 'true' : 'false'; ?>;
    </script>
    
    <?php
}

/**
 * Sauvegarde des meta données
 */
add_action('save_post_entity', function ($post_id) {
    
    if (!isset($_POST['eas_entity_nonce']) || !wp_verify_nonce($_POST['eas_entity_nonce'], 'eas_entity_meta')) {
        return;
    }
    
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }
    
    $fields = [
        'entity_canonical' => 'sanitize_text_field',
        'entity_url' => 'esc_url_raw',
        'entity_synonyms' => 'sanitize_textarea_field',
        'entity_image' => 'esc_url_raw',
        'entity_same_as' => 'sanitize_textarea_field',
        'entity_job_title' => 'sanitize_text_field',
        'entity_email' => 'sanitize_email',
        'entity_telephone' => 'sanitize_text_field',
        'entity_address_street' => 'sanitize_text_field',
        'entity_address_city' => 'sanitize_text_field',
        'entity_address_postal' => 'sanitize_text_field',
        'entity_address_country' => 'sanitize_text_field',
        'entity_works_for' => 'sanitize_text_field', // Peut être un ID ou 'geo_entities_org'
        'entity_member_of' => 'sanitize_text_field',
    ];
    
    foreach ($fields as $field => $sanitize_function) {
        if (isset($_POST[$field])) {
            $value = call_user_func($sanitize_function, $_POST[$field]);
            update_post_meta($post_id, '_' . $field, $value);
        }
    }
    
});