<?php
/**
 * Génération du JSON-LD pour les entités
 * V2.2 - CORRECTION COMPLÈTE
 * 
 * BUGS CORRIGÉS :
 * 1. "@type": "worksFor" → "@type": "Organization"
 * 2. worksFor avec @type → worksFor avec @id uniquement
 * 3. Doublons de <script> → Désactivé si GEO Entities actif
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Génère le JSON-LD pour toutes les entités du site
 * 
 * ⚠️ IMPORTANT : Si GEO Entities est actif, cette fonction ne fait rien
 * pour éviter les doublons de JSON-LD dans le HTML
 */
add_action('wp_head', function () {
    
    // ✅ CORRECTION : Si GEO Entities existe, laisser GEO générer le JSON-LD
    if (function_exists('geo_register_entity')) {
        // GEO Entities + Bridge s'occupent de tout
        return;
    }
    
    // Mode standalone : EAS génère son propre JSON-LD
    eas_generate_jsonld_standalone();
    
}, 30);

/**
 * Génération JSON-LD en mode standalone (sans GEO Entities)
 */
function eas_generate_jsonld_standalone() {
    
    $entities = get_posts([
        'post_type' => 'entity',
        'posts_per_page' => -1,
        'post_status' => 'publish',
    ]);
    
    if (empty($entities)) {
        return;
    }
    
    $graph = [];
    $processed_ids = [];
    
    foreach ($entities as $entity) {
        $entity_data = eas_build_entity_schema($entity);
        if ($entity_data && !in_array($entity_data['@id'], $processed_ids)) {
            $graph[] = $entity_data;
            $processed_ids[] = $entity_data['@id'];
        }
    }
    
    if (empty($graph)) {
        return;
    }
    
    echo "\n" . '<script type="application/ld+json">' . "\n";
    echo wp_json_encode(
        [
            '@context' => 'https://schema.org',
            '@graph' => $graph,
        ],
        JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT
    );
    echo "\n" . '</script>' . "\n";
}

/**
 * Construit le schema d'une entité selon son type
 */
function eas_build_entity_schema($entity) {
    
    $post_id = $entity->ID;
    
    // Récupérer le type d'entité
    $types = wp_get_post_terms($post_id, 'entity_type');
    if (!$types || is_wp_error($types)) {
        $type = 'Thing';
    } else {
        $type = $types[0]->name;
    }
    
    // Données de base
    $canonical = get_post_meta($post_id, '_entity_canonical', true);
    $name = !empty($canonical) ? $canonical : get_the_title($entity);
    $description = wp_strip_all_tags($entity->post_content);
    $url = get_post_meta($post_id, '_entity_url', true);
    
    // Générer un @id unique
    $id = home_url('/#entity-' . $post_id);
    
    // ✅ CORRECTION : Structure de base SANS @context ici (sera ajouté dans @graph)
    $schema = [
        '@type' => $type,
        '@id' => $id,
        'name' => $name,
    ];
    
    if (!empty($description)) {
        $schema['description'] = $description;
    }
    
    if (!empty($url)) {
        $schema['url'] = $url;
    }
    
    // Image / Logo
    $image = get_post_meta($post_id, '_entity_image', true);
    if (empty($image) && has_post_thumbnail($post_id)) {
        $image = get_the_post_thumbnail_url($post_id, 'full');
    }
    if (!empty($image)) {
        $schema['image'] = $image;
        if ($type === 'Organization') {
            $schema['logo'] = $image;
        }
    }
    
    // Synonymes (alternateName)
    $synonyms = get_post_meta($post_id, '_entity_synonyms', true);
    if (!empty($synonyms)) {
        $synonyms_array = array_map('trim', explode(',', $synonyms));
        if (count($synonyms_array) === 1) {
            $schema['alternateName'] = $synonyms_array[0];
        } else {
            $schema['alternateName'] = $synonyms_array;
        }
    }
    
    // SameAs (liens sociaux)
    $same_as = get_post_meta($post_id, '_entity_same_as', true);
    if (!empty($same_as)) {
        $same_as_array = array_filter(array_map('trim', explode("\n", $same_as)));
        if (!empty($same_as_array)) {
            $schema['sameAs'] = $same_as_array;
        }
    }
    
    // Propriétés spécifiques selon le type
    switch ($type) {
        case 'Person':
            $schema = eas_add_person_properties($schema, $post_id);
            break;
            
        case 'Organization':
        case 'LocalBusiness':
            $schema = eas_add_organization_properties($schema, $post_id);
            break;
    }
    
    return $schema;
}

/**
 * Ajoute les propriétés spécifiques à Person
 * 
 * ✅ CORRECTION MAJEURE : worksFor et memberOf sont maintenant de simples références @id
 */
function eas_add_person_properties($schema, $post_id) {
    
    $job_title = get_post_meta($post_id, '_entity_job_title', true);
    if (!empty($job_title)) {
        $schema['jobTitle'] = $job_title;
    }
    
    $email = get_post_meta($post_id, '_entity_email', true);
    if (!empty($email)) {
        $schema['email'] = $email;
    }
    
    $telephone = get_post_meta($post_id, '_entity_telephone', true);
    if (!empty($telephone)) {
        $schema['telephone'] = $telephone;
    }
    
    // ✅ CORRECTION : WorksFor doit être UNE SIMPLE RÉFÉRENCE, pas un objet complet
    $works_for = get_post_meta($post_id, '_entity_works_for', true);
    if (!empty($works_for)) {
        if ($works_for === 'geo_entities_org') {
            // Référencer l'Organization GEO Entities
            $schema['worksFor'] = [
                '@id' => home_url('/#organization'),
            ];
        } else {
            // Référencer une entité EAS (SANS @type !)
            $schema['worksFor'] = [
                '@id' => home_url('/#entity-' . $works_for),
            ];
        }
    }
    
    // ✅ CORRECTION : MemberOf aussi en référence simple
    $member_of = get_post_meta($post_id, '_entity_member_of', true);
    if (!empty($member_of)) {
        if ($member_of === 'geo_entities_org') {
            $schema['memberOf'] = [
                '@id' => home_url('/#organization'),
            ];
        } else {
            $schema['memberOf'] = [
                '@id' => home_url('/#entity-' . $member_of),
            ];
        }
    }
    
    return $schema;
}

/**
 * Ajoute les propriétés spécifiques à Organization/LocalBusiness
 */
function eas_add_organization_properties($schema, $post_id) {
    
    $email = get_post_meta($post_id, '_entity_email', true);
    if (!empty($email)) {
        $schema['email'] = $email;
    }
    
    $telephone = get_post_meta($post_id, '_entity_telephone', true);
    if (!empty($telephone)) {
        $schema['telephone'] = $telephone;
    }
    
    // Address
    $street = get_post_meta($post_id, '_entity_address_street', true);
    $city = get_post_meta($post_id, '_entity_address_city', true);
    $postal = get_post_meta($post_id, '_entity_address_postal', true);
    $country = get_post_meta($post_id, '_entity_address_country', true);
    
    if (!empty($street) || !empty($city)) {
        $address = [
            '@type' => 'PostalAddress',
        ];
        
        if (!empty($street)) $address['streetAddress'] = $street;
        if (!empty($city)) $address['addressLocality'] = $city;
        if (!empty($postal)) $address['postalCode'] = $postal;
        if (!empty($country)) $address['addressCountry'] = $country;
        
        $schema['address'] = $address;
    }
    
    return $schema;
}

/**
 * Ajoute le JSON-LD d'une entité sur les articles qui la mentionnent
 */
add_action('wp_head', function () {
    
    if (!is_singular(['post', 'page'])) {
        return;
    }
    
    // Si GEO Entities actif, ne rien faire
    if (function_exists('geo_register_entity')) {
        return;
    }
    
    $post_id = get_queried_object_id();
    $content = get_post_field('post_content', $post_id);
    
    preg_match_all('/\[entity id=(\d+)\]/', $content, $matches);
    
    if (empty($matches[1])) {
        return;
    }
    
    $entity_ids = array_unique($matches[1]);
    
    foreach ($entity_ids as $entity_id) {
        $entity = get_post($entity_id);
        if ($entity && $entity->post_type === 'entity') {
            $schema = eas_build_entity_schema($entity);
            if ($schema) {
                echo "\n" . '<script type="application/ld+json">' . "\n";
                echo wp_json_encode(
                    array_merge(['@context' => 'https://schema.org'], $schema),
                    JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT
                );
                echo "\n" . '</script>' . "\n";
            }
        }
    }
    
}, 35);

/**
 * Shortcode pour insérer une référence à une entité
 */
add_shortcode('entity', function ($atts) {
    $atts = shortcode_atts([
        'id' => 0,
    ], $atts);
    
    $entity_id = intval($atts['id']);
    if (!$entity_id) {
        return '';
    }
    
    $entity = get_post($entity_id);
    if (!$entity || $entity->post_type !== 'entity') {
        return '';
    }
    
    $canonical = get_post_meta($entity_id, '_entity_canonical', true);
    $name = !empty($canonical) ? $canonical : get_the_title($entity);
    $url = get_post_meta($entity_id, '_entity_url', true);
    
    if (!empty($url)) {
        return sprintf(
            '<a href="%s" class="entity-link" data-entity-id="%d">%s</a>',
            esc_url($url),
            $entity_id,
            esc_html($name)
        );
    }
    
    return sprintf(
        '<span class="entity-mention" data-entity-id="%d">%s</span>',
        $entity_id,
        esc_html($name)
    );
});