<?php
/**
 * Page d'administration pour gérer les doublons
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Ajoute la page de gestion des doublons
 */
add_action('admin_menu', function () {
    add_submenu_page(
        'edit.php?post_type=entity',
        'Vérification des doublons',
        'Doublons',
        'manage_options',
        'eas-duplicates',
        'eas_render_duplicates_page'
    );
});

/**
 * Affiche la page de gestion des doublons
 */
function eas_render_duplicates_page() {
    
    // Récupérer toutes les entités
    $entities = get_posts([
        'post_type' => 'entity',
        'posts_per_page' => -1,
        'post_status' => 'publish',
    ]);
    
    // Analyser les doublons
    $duplicates_groups = [];
    $checked = [];
    
    foreach ($entities as $entity) {
        if (in_array($entity->ID, $checked)) {
            continue;
        }
        
        $title = get_the_title($entity);
        $canonical = get_post_meta($entity->ID, '_entity_canonical', true);
        $search_name = !empty($canonical) ? $canonical : $title;
        
        $types = wp_get_post_terms($entity->ID, 'entity_type');
        $type = $types && !is_wp_error($types) ? $types[0]->name : '';
        
        // Chercher les doublons
        $duplicates = eas_find_duplicate_entities($search_name, $type, $entity->ID);
        
        if (!empty($duplicates)) {
            $duplicates_groups[] = [
                'main' => [
                    'id' => $entity->ID,
                    'title' => $title,
                    'type' => $type,
                    'url' => get_post_meta($entity->ID, '_entity_url', true),
                ],
                'duplicates' => $duplicates,
            ];
            
            $checked[] = $entity->ID;
            foreach ($duplicates as $dup) {
                if ($dup['id']) {
                    $checked[] = $dup['id'];
                }
            }
        }
    }
    
    ?>
    <div class="wrap">
        <h1>🔍 Vérification des doublons d'entités</h1>
        
        <p class="description">
            Cette page détecte les entités en doublon pour éviter les conflits dans le Schema.org. 
            Les doublons peuvent provenir d'Entity Authority Signals, GEO Entities, ou des auteurs WordPress.
        </p>
        
        <?php if (empty($duplicates_groups)): ?>
            
            <div class="notice notice-success" style="margin-top: 20px;">
                <p><strong>✅ Aucun doublon détecté !</strong></p>
                <p>Toutes vos entités semblent uniques.</p>
            </div>
            
        <?php else: ?>
            
            <div class="notice notice-warning" style="margin-top: 20px;">
                <p><strong>⚠️ <?php echo count($duplicates_groups); ?> groupe(s) de doublons détecté(s)</strong></p>
                <p>Vérifiez les doublons ci-dessous et supprimez les entités en trop.</p>
            </div>
            
            <?php foreach ($duplicates_groups as $index => $group): ?>
                
                <div class="card" style="margin-top: 20px;">
                    <h2 style="margin-top: 0;">Groupe #<?php echo ($index + 1); ?></h2>
                    
                    <!-- Entité principale -->
                    <div style="padding: 15px; background: #e7f3ff; border-left: 4px solid #0073aa; margin-bottom: 15px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <div>
                                <strong style="font-size: 16px;">
                                    <?php echo esc_html($group['main']['title']); ?>
                                </strong>
                                
                                <?php if ($group['main']['type']): ?>
                                    <span style="background: #0073aa; color: white; padding: 3px 8px; border-radius: 3px; font-size: 12px; margin-left: 10px;">
                                        <?php echo esc_html($group['main']['type']); ?>
                                    </span>
                                <?php endif; ?>
                                
                                <?php if ($group['main']['url']): ?>
                                    <div style="margin-top: 5px; font-size: 13px; color: #555;">
                                        🔗 <a href="<?php echo esc_url($group['main']['url']); ?>" target="_blank">
                                            <?php echo esc_html($group['main']['url']); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                                
                                <div style="margin-top: 5px; font-size: 12px; color: #666;">
                                    Source: Entity Authority Signals • ID: <?php echo $group['main']['id']; ?>
                                </div>
                            </div>
                            
                            <div>
                                <a href="<?php echo get_edit_post_link($group['main']['id']); ?>" 
                                   class="button button-primary">
                                    Modifier
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Doublons -->
                    <h3 style="margin-top: 20px; color: #d63638;">⚠️ Doublons trouvés :</h3>
                    
                    <?php foreach ($group['duplicates'] as $duplicate): ?>
                        <div style="padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107; margin-bottom: 10px;">
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <div>
                                    <strong><?php echo esc_html($duplicate['title']); ?></strong>
                                    
                                    <?php if ($duplicate['type']): ?>
                                        <span style="background: #856404; color: white; padding: 3px 8px; border-radius: 3px; font-size: 12px; margin-left: 10px;">
                                            <?php echo esc_html($duplicate['type']); ?>
                                        </span>
                                    <?php endif; ?>
                                    
                                    <?php if ($duplicate['url']): ?>
                                        <div style="margin-top: 5px; font-size: 13px;">
                                            🔗 <a href="<?php echo esc_url($duplicate['url']); ?>" target="_blank">
                                                <?php echo esc_html($duplicate['url']); ?>
                                            </a>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div style="margin-top: 5px; font-size: 12px; color: #856404;">
                                        Source: <?php echo esc_html($duplicate['source']); ?>
                                        <?php if ($duplicate['id']): ?>
                                            • ID: <?php echo $duplicate['id']; ?>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <div>
                                    <?php if ($duplicate['source'] === 'Entity Authority Signals' && $duplicate['id']): ?>
                                        <a href="<?php echo get_edit_post_link($duplicate['id']); ?>" 
                                           class="button button-secondary"
                                           target="_blank">
                                            Voir
                                        </a>
                                        <a href="<?php echo get_delete_post_link($duplicate['id'], '', true); ?>" 
                                           class="button button-link-delete"
                                           onclick="return confirm('Êtes-vous sûr de vouloir supprimer cette entité ?');">
                                            Supprimer
                                        </a>
                                    <?php else: ?>
                                        <span style="color: #666; font-size: 12px;">
                                            (Géré par <?php echo esc_html($duplicate['source']); ?>)
                                        </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    
                    <div style="margin-top: 15px; padding: 10px; background: #f0f0f0; border-radius: 4px;">
                        <strong>💡 Recommandation :</strong>
                        Gardez une seule entité et supprimez les doublons. Si vous supprimez l'entité principale, 
                        assurez-vous de mettre à jour les relations (worksFor, memberOf) dans les autres entités.
                    </div>
                </div>
                
            <?php endforeach; ?>
            
        <?php endif; ?>
        
        <div class="card" style="margin-top: 30px;">
            <h2>📋 Statistiques</h2>
            <ul>
                <li><strong>Total entités :</strong> <?php echo count($entities); ?></li>
                <li><strong>Groupes de doublons :</strong> <?php echo count($duplicates_groups); ?></li>
                <li><strong>Entités uniques :</strong> <?php echo count($entities) - count($checked); ?></li>
            </ul>
        </div>
        
        <p style="margin-top: 20px;">
            <a href="<?php echo admin_url('edit.php?post_type=entity'); ?>" class="button">
                ← Retour aux entités
            </a>
            <button onclick="location.reload();" class="button button-primary">
                🔄 Revérifier les doublons
            </button>
        </p>
    </div>
    <?php
}