<?php
/**
 * Plugin Name: Entity & Authority Signals V2
 * Description: Centralise les entités clés d'un site WordPress et expose des signaux d'autorité lisibles par les moteurs IA. Optimisé pour le GEO (Generative Engine Optimization).
 * Version: 2.0.0
 * Author: Erwan Tanguy - Ticoët
 * License: GPL2+
 * Text Domain: entity-authority-signals
 */

if (!defined('ABSPATH')) {
    exit;
}

define('EAS_VERSION', '2.0.0');
define('EAS_PATH', plugin_dir_path(__FILE__));
define('EAS_URL', plugin_dir_url(__FILE__));

// Chargement des composants
require_once EAS_PATH . 'includes/cpt-entity.php';
require_once EAS_PATH . 'includes/taxonomy-entity-type.php';
require_once EAS_PATH . 'includes/meta-boxes.php';
require_once EAS_PATH . 'includes/jsonld-output.php';
require_once EAS_PATH . 'includes/duplicate-detection.php'; // NOUVEAU
require_once EAS_PATH . 'includes/admin-duplicates-page.php'; // NOUVEAU

/**
 * Activation du plugin
 */
register_activation_hook(__FILE__, function () {
    // Forcer la création des types et taxonomies
    do_action('init');
    
    // Flush rewrite rules
    flush_rewrite_rules();
    
    // Message de bienvenue
    set_transient('eas_activation_notice', true, 5);
});

/**
 * Désactivation du plugin
 */
register_deactivation_hook(__FILE__, function () {
    flush_rewrite_rules();
});

/**
 * Message après activation
 */
add_action('admin_notices', function () {
    if (get_transient('eas_activation_notice')) {
        ?>
        <div class="notice notice-success is-dismissible">
            <p><strong>Entity & Authority Signals V2 activé !</strong></p>
            <p>Vous pouvez maintenant créer vos entités dans le menu <strong>Entités</strong>.</p>
            <p>Les types Schema.org courants ont été pré-créés : Organization, Person, LocalBusiness, etc.</p>
        </div>
        <?php
        delete_transient('eas_activation_notice');
    }
});

/**
 * Ajoute un menu d'aide
 */
add_action('admin_menu', function () {
    add_submenu_page(
        'edit.php?post_type=entity',
        'Aide & Documentation',
        'Aide',
        'manage_options',
        'eas-help',
        'eas_render_help_page'
    );
});

/**
 * Page d'aide
 */
function eas_render_help_page() {
    ?>
    <div class="wrap">
        <h1>📚 Entity & Authority Signals - Aide</h1>
        
        <div class="card">
            <h2>🎯 Objectif du plugin</h2>
            <p>Ce plugin permet de centraliser toutes les entités importantes de votre site (personnes, organisations, produits, etc.) et de les exposer au format Schema.org pour :</p>
            <ul>
                <li><strong>Améliorer votre SEO</strong> (Google comprend mieux votre contenu)</li>
                <li><strong>Optimiser votre GEO</strong> (Generative Engine Optimization pour ChatGPT, Perplexity, etc.)</li>
                <li><strong>Être cité comme source</strong> par les moteurs IA</li>
            </ul>
        </div>
        
        <div class="card">
            <h2>🚀 Comment l'utiliser</h2>
            
            <h3>1. Créer une Organization principale</h3>
            <ol>
                <li>Aller dans <strong>Entités > Ajouter</strong></li>
                <li>Titre : Le nom de votre entreprise/site</li>
                <li>Type : <strong>Organization</strong></li>
                <li>Remplir les champs : URL, description, logo, adresse</li>
                <li>Ajouter les liens sociaux dans "sameAs"</li>
            </ol>
            
            <h3>2. Créer les Person (auteurs, employés)</h3>
            <ol>
                <li>Créer une nouvelle entité pour chaque personne</li>
                <li>Type : <strong>Person</strong></li>
                <li>Remplir : Fonction, email, photo</li>
                <li><strong>Important :</strong> Dans "Relations", sélectionner l'Organization dans "Travaille pour"</li>
            </ol>
            
            <h3>3. Mentionner les entités dans vos articles</h3>
            <p>Utilisez le shortcode <code>[entity id=X]</code> pour mentionner une entité :</p>
            <pre>J'ai rencontré [entity id=5] lors de la conférence...</pre>
            <p>L'entité sera automatiquement incluse dans le JSON-LD de l'article.</p>
        </div>
        
        <div class="card">
            <h2>🔍 Vérifier le JSON-LD généré</h2>
            <ol>
                <li>Afficher le code source de votre page (Ctrl+U)</li>
                <li>Chercher <code>&lt;script type="application/ld+json"&gt;</code></li>
                <li>Tester sur <a href="https://validator.schema.org/" target="_blank">Schema.org Validator</a></li>
            </ol>
        </div>
        
        <div class="card">
            <h2>💡 Bonnes pratiques GEO</h2>
            <ul>
                <li>✅ <strong>Une seule Organization principale</strong> pour votre site</li>
                <li>✅ <strong>Relier toutes les Person</strong> à cette Organization via "worksFor"</li>
                <li>✅ <strong>Ajouter des photos/logos</strong> pour chaque entité</li>
                <li>✅ <strong>Remplir les descriptions</strong> avec soin</li>
                <li>✅ <strong>Ajouter les liens sociaux</strong> (Facebook, LinkedIn, Twitter)</li>
                <li>✅ <strong>Utiliser le shortcode [entity]</strong> pour créer des liens entre contenus et entités</li>
            </ul>
        </div>
        
        <div class="card">
            <h2>📊 Types d'entités disponibles</h2>
            <ul>
                <li><strong>Organization</strong> : Votre entreprise, association</li>
                <li><strong>Person</strong> : Auteurs, employés, experts</li>
                <li><strong>LocalBusiness</strong> : Entreprise avec adresse physique</li>
                <li><strong>Product</strong> : Produits que vous vendez</li>
                <li><strong>Service</strong> : Services que vous proposez</li>
                <li><strong>Place</strong> : Lieux géographiques</li>
                <li><strong>Event</strong> : Événements</li>
            </ul>
            <p>Vous pouvez ajouter d'autres types dans <strong>Entités > Types d'entités</strong>.</p>
        </div>
        
        <div class="card">
            <h2>🔗 Ressources</h2>
            <ul>
                <li><a href="https://schema.org/" target="_blank">Schema.org Documentation</a></li>
                <li><a href="https://validator.schema.org/" target="_blank">Schema.org Validator</a></li>
                <li><a href="https://search.google.com/test/rich-results" target="_blank">Google Rich Results Test</a></li>
            </ul>
        </div>
    </div>
    <?php
}

/**
 * Styles admin
 */
add_action('admin_head', function () {
    $screen = get_current_screen();
    if ($screen && $screen->post_type === 'entity') {
        ?>
        <style>
            .eas-schema-properties {
                background: #fff;
            }
            .eas-schema-properties h4 {
                margin: 20px 0 10px;
                padding: 10px;
                background: #f0f0f0;
                border-left: 3px solid #0073aa;
                font-size: 14px;
                font-weight: 600;
            }
            .eas-schema-properties .form-table th {
                width: 200px;
            }
        </style>
        <?php
    }
});